from flask import Flask, request, jsonify, render_template, send_from_directory
from flask_cors import CORS
import os
import json
import logging
from datetime import datetime, timedelta
import sqlite3
from typing import List, Dict, Any
import hashlib
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)
app = Flask(__name__)
CORS(app)
DATABASE_PATH = '/app/data/analytics.db'
DATA_DIR = '/app/data'
os.makedirs(DATA_DIR, exist_ok=True)
def init_database():

    try:
        logger.info(f"Инициализация базы данных: {DATABASE_PATH}")
        logger.info(f"Директория данных: {DATA_DIR}")
        conn = sqlite3.connect(DATABASE_PATH)
        cursor = conn.cursor()
        cursor.execute()
        cursor.execute()
        cursor.execute('CREATE INDEX IF NOT EXISTS idx_conversations_id ON conversations (conversation_id)')
        cursor.execute('CREATE INDEX IF NOT EXISTS idx_messages_conversation ON messages (conversation_id)')
        cursor.execute('CREATE INDEX IF NOT EXISTS idx_messages_timestamp ON messages (timestamp)')
        try:
            cursor.execute('ALTER TABLE messages ADD COLUMN llm_generation_time REAL')
            logger.info("Добавлена колонка llm_generation_time")
        except sqlite3.OperationalError as e:
            if "duplicate column name" in str(e):
                logger.info("Колонка llm_generation_time уже существует")
            else:
                logger.error(f"Ошибка добавления колонки: {e}")
        conn.commit()
        conn.close()
        logger.info("База данных аналитики инициализирована")
    except Exception as e:
        logger.error(f"Ошибка инициализации базы данных: {e}")
def generate_conversation_id(session_data: Dict[str, Any]) -> str:

    ip = session_data.get('ip_address', 'unknown')
    ua = session_data.get('user_agent', 'unknown')
    timestamp = datetime.now().strftime('%Y%m%d%H')
    raw_id = f"{ip}_{ua}_{timestamp}"
    return hashlib.md5(raw_id.encode()).hexdigest()[:16]
@app.route('/')
def index():

    return render_template('analytics-simple.html')
@app.route('/test')
def test():

    return render_template('test.html')
@app.route('/static/<path:filename>')
def static_files(filename):

    return send_from_directory('/app/static', filename)
@app.route('/api/conversations')
def get_conversations():

    try:
        conn = sqlite3.connect(DATABASE_PATH)
        cursor = conn.cursor()
        cursor.execute('SELECT COUNT(*) FROM conversations')
        total_conversations = cursor.fetchone()[0]
        if total_conversations == 0:
            conn.close()
            return jsonify({
                'status': 'success',
                'conversations': [],
                'total': 0
            })
        cursor.execute()
        conversations = []
        for row in cursor.fetchall():
            conversations.append({
                'id': row[0],
                'conversation_id': row[1],
                'session_id': row[2],
                'ip_address': row[3],
                'created_at': row[4],
                'updated_at': row[5],
                'message_count': row[6],
                'last_message': row[9] if row[9] else row[7],
                'status': row[8],
                'actual_message_count': row[6]
            })
        conn.close()
        return jsonify({
            'status': 'success',
            'conversations': conversations,
            'total': len(conversations)
        })
    except Exception as e:
        logger.error(f"Ошибка получения диалогов: {e}")
        return jsonify({'error': 'Ошибка получения диалогов'}), 500
@app.route('/api/conversations/<conversation_id>')
def get_conversation_detail(conversation_id):

    try:
        conn = sqlite3.connect(DATABASE_PATH)
        cursor = conn.cursor()
        cursor.execute('''
            SELECT 
                conversation_id,
                session_id,
                user_agent,
                ip_address,
                created_at,
                updated_at,
                message_count,
                last_message,
                status
            FROM conversations 
            WHERE conversation_id = ?
        ''', (conversation_id,))
        conversation_data = cursor.fetchone()
        if not conversation_data:
            return jsonify({'error': 'Диалог не найден'}), 404
        cursor.execute('''
            SELECT 
                message_type,
                content,
                timestamp,
                metadata,
                llm_generation_time
            FROM messages 
            WHERE conversation_id = ?
            ORDER BY timestamp ASC
        ''', (conversation_id,))
        messages = []
        for row in cursor.fetchall():
            messages.append({
                'type': row[0],
                'content': row[1],
                'timestamp': row[2],
                'metadata': json.loads(row[3]) if row[3] else {},
                'llm_generation_time': row[4]
            })
        conn.close()
        return jsonify({
            'status': 'success',
            'conversation': {
                'conversation_id': conversation_data[0],
                'session_id': conversation_data[1],
                'user_agent': conversation_data[2],
                'ip_address': conversation_data[3],
                'created_at': conversation_data[4],
                'updated_at': conversation_data[5],
                'message_count': conversation_data[6],
                'last_message': conversation_data[7],
                'status': conversation_data[8],
                'messages': messages
            }
        })
    except Exception as e:
        logger.error(f"Ошибка получения диалога {conversation_id}: {e}")
        return jsonify({'error': 'Ошибка получения диалога'}), 500
@app.route('/api/conversations/<conversation_id>/messages', methods=['POST'])
def add_message(conversation_id):

    try:
        data = request.get_json()
        message_type = data.get('type', 'user')
        content = data.get('content', '')
        metadata = data.get('metadata', {})
        llm_generation_time = data.get('llm_generation_time')
        logger.info(f"Получено сообщение: type={message_type}, llm_generation_time={llm_generation_time}")
        if not content:
            return jsonify({'error': 'Содержимое сообщения не может быть пустым'}), 400
        conn = sqlite3.connect(DATABASE_PATH)
        cursor = conn.cursor()
        cursor.execute('SELECT id FROM conversations WHERE conversation_id = ?', (conversation_id,))
        if not cursor.fetchone():
            real_ip = (request.headers.get('X-Real-IP') or
                      request.headers.get('X-Forwarded-For', '').split(',')[0].strip() or
                      request.remote_addr)
            session_data = {
                'ip_address': real_ip,
                'user_agent': request.headers.get('User-Agent', '')
            }
            cursor.execute('''
                INSERT INTO conversations 
                (conversation_id, session_id, user_agent, ip_address, last_message, message_count)
                VALUES (?, ?, ?, ?, ?, 1)
            ''', (
                conversation_id,
                data.get('session_id', ''),
                session_data['user_agent'],
                session_data['ip_address'],
                content[:100]
            ))
        else:
            cursor.execute('''
                UPDATE conversations 
                SET 
                    updated_at = CURRENT_TIMESTAMP,
                    message_count = message_count + 1,
                    last_message = ?
                WHERE conversation_id = ?
            ''', (content[:100], conversation_id))
        cursor.execute('''
            INSERT INTO messages (conversation_id, message_type, content, metadata, llm_generation_time)
            VALUES (?, ?, ?, ?, ?)
        ''', (conversation_id, message_type, content, json.dumps(metadata), llm_generation_time))
        conn.commit()
        conn.close()
        return jsonify({
            'status': 'success',
            'message': 'Сообщение добавлено'
        })
    except Exception as e:
        logger.error(f"Ошибка добавления сообщения: {e}")
        return jsonify({'error': 'Ошибка добавления сообщения'}), 500
@app.route('/api/test-time', methods=['POST'])
def test_generation_time():

    try:
        data = request.get_json()
        print(f"TEST: Получены данные: {data}")
        conversation_id = data.get('conversation_id', 'test')
        message_type = data.get('type', 'assistant')
        content = data.get('content', 'test message')
        llm_generation_time = data.get('llm_generation_time')
        print(f"TEST: conversation_id={conversation_id}, type={message_type}, time={llm_generation_time}")
        conn = sqlite3.connect(DATABASE_PATH)
        cursor = conn.cursor()
        cursor.execute('''
            INSERT INTO messages (conversation_id, message_type, content, metadata, llm_generation_time)
            VALUES (?, ?, ?, ?, ?)
        ''', (conversation_id, message_type, content, '{}', llm_generation_time))
        conn.commit()
        conn.close()
        print(f"TEST: Сохранено в базу: time={llm_generation_time}")
        return jsonify({
            'status': 'success',
            'message': 'Тестовое сообщение сохранено',
            'llm_generation_time': llm_generation_time
        })
    except Exception as e:
        print(f"TEST ERROR: {e}")
        return jsonify({'error': str(e)}), 500
@app.route('/api/stats')
def get_stats():

    try:
        conn = sqlite3.connect(DATABASE_PATH)
        cursor = conn.cursor()
        cursor.execute('SELECT COUNT(*) FROM conversations')
        total_conversations = cursor.fetchone()[0]
        cursor.execute('SELECT COUNT(*) FROM messages')
        total_messages = cursor.fetchone()[0]
        cursor.execute('SELECT COUNT(*) FROM messages WHERE message_type = "user"')
        user_messages = cursor.fetchone()[0]
        cursor.execute('SELECT COUNT(*) FROM messages WHERE message_type = "assistant"')
        assistant_messages = cursor.fetchone()[0]
        cursor.execute()
        conversations_today = cursor.fetchone()[0]
        cursor.execute()
        messages_today = cursor.fetchone()[0]
        cursor.execute()
        top_ips = [{'ip': row[0], 'count': row[1]} for row in cursor.fetchall()]
        cursor.execute('SELECT COUNT(DISTINCT ip_address) FROM conversations')
        unique_users = cursor.fetchone()[0]
        avg_messages_per_conversation = round(total_messages / total_conversations, 1) if total_conversations > 0 else 0
        cursor.execute()
        llm_timing = cursor.fetchone()
        avg_generation_time = round(llm_timing[0], 2) if llm_timing[0] else 0
        min_generation_time = round(llm_timing[1], 2) if llm_timing[1] else 0
        max_generation_time = round(llm_timing[2], 2) if llm_timing[2] else 0
        messages_with_timing = llm_timing[3] if llm_timing[3] else 0
        avg_response_length = round(llm_timing[4], 0) if llm_timing[4] else 0
        min_response_length = llm_timing[5] if llm_timing[5] else 0
        max_response_length = llm_timing[6] if llm_timing[6] else 0
        avg_time_per_char = round(llm_timing[7], 4) if llm_timing[7] else 0
        fast_responses = llm_timing[8] if llm_timing[8] else 0
        medium_responses = llm_timing[9] if llm_timing[9] else 0
        slow_responses = llm_timing[10] if llm_timing[10] else 0
        cursor.execute()
        hourly_stats = [{'hour': row[0], 'count': row[1], 'avg_time': round(row[2], 2)} for row in cursor.fetchall()]
        cursor.execute()
        daily_stats = [{'day': row[0], 'count': row[1], 'avg_time': round(row[2], 2)} for row in cursor.fetchall()]
        cursor.execute()
        length_stats = [{'category': row[0], 'count': row[1], 'avg_time': round(row[2], 2)} for row in cursor.fetchall()]
        cursor.execute()
        quality_metrics = cursor.fetchone()
        avg_length = round(quality_metrics[0], 0) if quality_metrics[0] else 0
        avg_word_count = round(quality_metrics[1], 1) if quality_metrics[1] else 0
        avg_sentence_count = round(quality_metrics[2], 1) if quality_metrics[2] else 0
        formatted_responses = quality_metrics[3] if quality_metrics[3] else 0
        structured_responses = quality_metrics[4] if quality_metrics[4] else 0
        questions_in_response = quality_metrics[5] if quality_metrics[5] else 0
        efficiency_score = round(quality_metrics[6], 4) if quality_metrics[6] else 0
        cursor.execute()
        quality_categories = [{'category': row[0], 'count': row[1], 'avg_time': round(row[2], 2), 'avg_words': round(row[3], 1)} for row in cursor.fetchall()]
        conn.close()
        return jsonify({
            'status': 'success',
            'stats': {
                'total_conversations': total_conversations,
                'total_messages': total_messages,
                'user_messages': user_messages,
                'assistant_messages': assistant_messages,
                'conversations_today': conversations_today,
                'messages_today': messages_today,
                'unique_users': unique_users,
                'avg_messages_per_conversation': avg_messages_per_conversation,
                'top_ips': top_ips,
                'llm_generation_time': {
                    'avg_seconds': avg_generation_time,
                    'min_seconds': min_generation_time,
                    'max_seconds': max_generation_time,
                    'messages_with_timing': messages_with_timing
                },
                'llm_performance': {
                    'avg_response_length': avg_response_length,
                    'min_response_length': min_response_length,
                    'max_response_length': max_response_length,
                    'avg_time_per_char': avg_time_per_char,
                    'fast_responses': fast_responses,
                    'medium_responses': medium_responses,
                    'slow_responses': slow_responses
                },
                'llm_timing_patterns': {
                    'hourly_stats': hourly_stats,
                    'daily_stats': daily_stats,
                    'length_stats': length_stats
                },
                'llm_quality_metrics': {
                    'avg_length': avg_length,
                    'avg_word_count': avg_word_count,
                    'avg_sentence_count': avg_sentence_count,
                    'formatted_responses': formatted_responses,
                    'structured_responses': structured_responses,
                    'questions_in_response': questions_in_response,
                    'efficiency_score': efficiency_score,
                    'quality_categories': quality_categories
                }
            }
        })
    except Exception as e:
        logger.error(f"Ошибка получения статистики: {e}")
        return jsonify({'error': 'Ошибка получения статистики'}), 500
if __name__ == '__main__':
    init_database()
    app.run(host='0.0.0.0', port=9005, debug=True)