#!/bin/bash

set -e

# Получаем директорию где находится скрипт
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
cd "$SCRIPT_DIR"

echo "=== Настройка окружения для Jupyter Notebooks ==="
echo "Рабочая директория: $SCRIPT_DIR"

# Проверяем наличие Python
if ! command -v python3 &> /dev/null; then
    echo "Ошибка: Python3 не найден. Установите Python 3.9 или новее."
    exit 1
fi

# Проверяем наличие requirements.txt
if [ ! -f "requirements.txt" ]; then
    echo "Ошибка: файл requirements.txt не найден в текущей директории."
    exit 1
fi

# Создаем виртуальное окружение если его нет
if [ ! -d "venv" ]; then
    echo "Создание виртуального окружения..."
    python3 -m venv venv
else
    echo "Виртуальное окружение уже существует."
fi

# Активируем виртуальное окружение
echo "Активация виртуального окружения..."
source venv/bin/activate

# Обновляем pip
echo "Обновление pip..."
pip install --upgrade pip --quiet

# Устанавливаем зависимости
echo "Установка зависимостей из requirements.txt..."
pip install -r requirements.txt

# Регистрируем Python kernel для Jupyter
echo "Регистрация Python kernel..."
python3 -m ipykernel install --user --name=rag-chat-api --display-name "Python (RAG Chat API)"

echo ""
echo "=== Окружение готово ==="
echo ""

# Устанавливаем простой пароль для примеров (можно изменить)
JUPYTER_PASSWORD="jupyter"

echo "Запуск Jupyter Notebook..."
echo "Пароль для доступа: $JUPYTER_PASSWORD"
echo "Для остановки нажмите Ctrl+C"
echo ""

# Генерируем хеш пароля
PASSWORD_HASH=$(python3 <<EOF
try:
    from jupyter_server.auth import passwd
except ImportError:
    from notebook.auth import passwd
print(passwd('$JUPYTER_PASSWORD'))
EOF
)

# Запускаем Jupyter с паролем
echo ""
echo "Jupyter Notebook запускается..."
echo "После запуска открой в браузере:"
echo "  http://localhost:8888"
echo ""
echo "Пароль: $JUPYTER_PASSWORD"
echo ""

exec jupyter notebook --allow-root --ip=0.0.0.0 --NotebookApp.password="$PASSWORD_HASH" --NotebookApp.token='' --NotebookApp.open_browser=False
