from flask import Flask, request, jsonify, render_template, send_from_directory, Response
from flask_cors import CORS
import os
import json
import sys
import logging
import requests
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry
import time
from datetime import datetime
from typing import List, Dict, Any, Optional
from functools import lru_cache
import hashlib
import uuid
import re
from werkzeug.utils import secure_filename
from werkzeug.exceptions import RequestEntityTooLarge
from embedding_service import embedding_service
import threading
from collections import defaultdict
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)
app = Flask(__name__)
CORS(app)
app.config['MAX_CONTENT_LENGTH'] = 500 * 1024 * 1024
crawler_tasks = {}
crawler_tasks_lock = threading.Lock()
OLLAMA_BASE_URL = os.getenv('OLLAMA_BASE_URL', '')
OLLAMA_BASE_URL_FIRST = OLLAMA_BASE_URL.split(',')[0].strip() if OLLAMA_BASE_URL else ''
CHROMADB_BASE_URL = os.getenv('CHROMADB_BASE_URL', '')
ANALYTICS_BASE_URL = os.getenv('ANALYTICS_BASE_URL', '')
MODEL_NAME = os.getenv('MODEL_NAME', '')
EMBEDDING_MODEL = os.getenv('EMBEDDING_MODEL', '')
TEMPERATURE = float(os.getenv('TEMPERATURE', '0.2'))
NUM_CTX = int(os.getenv('NUM_CTX', '50000'))
SEARCH_K = int(os.getenv('SEARCH_K', '100'))
TIMEOUT = int(os.getenv('TIMEOUT', '600'))
NUM_PREDICT = int(os.getenv('NUM_PREDICT', '20000'))
REPEAT_PENALTY = float(os.getenv('REPEAT_PENALTY', '1.2'))
THINK_LEVEL = os.getenv('THINK_LEVEL', 'false')
if THINK_LEVEL.lower() in ('true', '1', 'yes'):
    THINK_LEVEL = True
elif THINK_LEVEL.lower() in ('false', '0', 'no', ''):
    THINK_LEVEL = False
STOP_WORDS_STR = os.getenv('STOP_WORDS', '')
try:
    if STOP_WORDS_STR:
        STOP_WORDS = json.loads(STOP_WORDS_STR)
        if not isinstance(STOP_WORDS, list):
            raise ValueError("STOP_WORDS must be a JSON array")
        STOP_WORDS = [str(word) for word in STOP_WORDS]
        logger.info(f"STOP_WORDS загружены из переменной окружения: {[repr(w) for w in STOP_WORDS]}")
    else:
        STOP_WORDS = ["\n\n\n\n"]
        logger.info(f"STOP_WORDS используют значения по умолчанию: {[repr(w) for w in STOP_WORDS]}")
except (json.JSONDecodeError, TypeError, ValueError) as e:
    logger.warning(f"Ошибка парсинга STOP_WORDS '{STOP_WORDS_STR}': {e}. Используются значения по умолчанию.")
    STOP_WORDS = ["\n\n\n\n"]
def parse_keep_alive(value: Optional[str]) -> Any:

    default = '12h'
    if value is None:
        return default
    value = str(value).strip()
    if not value:
        return default
    if value.isdigit():
        return int(value)
    try:
        float_val = float(value)
        return float_val
    except ValueError:
        pass
    return value
KEEP_ALIVE = parse_keep_alive(os.getenv('OLLAMA_KEEP_ALIVE', '12h'))
ollama_session = requests.Session()
retry_strategy = Retry(
    total=3,
    backoff_factor=1,
    status_forcelist=[429, 500, 502, 503, 504],
)
pool_connections = int(os.getenv('OLLAMA_POOL_CONNECTIONS', '20'))
pool_maxsize = int(os.getenv('OLLAMA_POOL_MAXSIZE', '100'))
adapter = HTTPAdapter(
    max_retries=retry_strategy,
    pool_connections=pool_connections,
    pool_maxsize=pool_maxsize
)
logger.info(f"Connection pool настроен: pool_connections={pool_connections}, pool_maxsize={pool_maxsize}")
ollama_session.mount("http://", adapter)
ollama_session.mount("https://", adapter)
ollama_session.headers.update({
    'Connection': 'keep-alive',
    'Keep-Alive': 'timeout=300, max=1000'
})
logger.info("Ollama session с connection pooling настроена")
logger.info(f"Ollama keep_alive установлен на {KEEP_ALIVE!r} ({'секунды' if isinstance(KEEP_ALIVE, (int, float)) else 'интервал'})")
PROMPT_CONTEXT_GUARD_RATIO = 0.75
def sanitize_prompt_section(value: Optional[str]) -> str:

    if value is None:
        return ""
    text = str(value)
    text = text.replace('</s>', ' ').replace('<s>', ' ')
    return text.replace('{', '{{').replace('}', '}}')
def enforce_context_limit(text: str, context_label: str = "context", max_chars: Optional[int] = None) -> str:

    if max_chars is None:
        max_tokens = int(NUM_CTX * PROMPT_CONTEXT_GUARD_RATIO)
        max_chars = max_tokens * 4
    if len(text) > max_chars:
        original_len = len(text)
        truncated = text[:max_chars]
        logger.warning(f"{context_label} слишком большой ({original_len} chars, ~{original_len//4} токенов). Обрезаю до {max_chars} chars (~{max_chars//4} токенов)")
        logger.debug(f"   Обрезано: {original_len - max_chars} символов (~{(original_len - max_chars)//4} токенов)")
        return truncated
    return text
class PerformanceMonitor:

    def __init__(self):
        self.metrics = {}
    def record_request(self, operation: str, duration: float, success: bool):

        if operation not in self.metrics:
            self.metrics[operation] = []
        self.metrics[operation].append({
            'duration': duration,
            'success': success,
            'timestamp': time.time()
        })
        if len(self.metrics[operation]) > 1000:
            self.metrics[operation] = self.metrics[operation][-500:]
    def get_stats(self, operation: str) -> Dict[str, Any]:

        if operation not in self.metrics:
            return {'count': 0, 'avg_duration': 0, 'success_rate': 0}
        metrics = self.metrics[operation]
        if not metrics:
            return {'count': 0, 'avg_duration': 0, 'success_rate': 0}
        durations = [m['duration'] for m in metrics]
        successes = [m['success'] for m in metrics]
        return {
            'count': len(metrics),
            'avg_duration': sum(durations) / len(durations),
            'success_rate': sum(successes) / len(successes) * 100
        }
perf_monitor = PerformanceMonitor()
@lru_cache(maxsize=1000)
def get_embedding_cached(text: str) -> List[float]:

    return embedding_service.get_embedding_cached(text)
CUSTOMER_PROMPT = ""
CUSTOMER_NOTES = ""
class RAGChatAPI:

    def __init__(self):
        self.conversations = {}
        self.logger = logger
    def generate_conversation_id(self, session_data: Dict[str, Any]) -> str:

        ip = session_data.get('ip_address', 'unknown')
        ua = session_data.get('user_agent', 'unknown')
        timestamp = datetime.now().strftime('%Y%m%d%H')
        raw_id = f"{ip}_{ua}_{timestamp}"
        return hashlib.md5(raw_id.encode()).hexdigest()[:16]
    def get_conversation_context(self, conversation_id: str) -> str:

        if conversation_id not in self.conversations:
            self.conversations[conversation_id] = []
        user_name = None
        valid_names = {
            'александр', 'алексей', 'анатолий', 'андрей', 'антон', 'артем', 'артём', 'борис', 'вадим', 'валентин',
            'валерий', 'василий', 'виктор', 'владимир', 'владислав', 'всеволод', 'геннадий', 'георгий', 'григорий',
            'даниил', 'данил', 'денис', 'дмитрий', 'евгений', 'егор', 'иван', 'игорь', 'илья', 'кирилл', 'константин',
            'максим', 'михаил', 'николай', 'олег', 'павел', 'петр', 'пётр', 'роман', 'сергей', 'станислав', 'степан',
            'тимофей', 'федор', 'фёдор', 'юрий', 'ярослав',
            'александра', 'анастасия', 'анна', 'валентина', 'валерия', 'вера', 'виктория', 'галина', 'дарья', 'елена',
            'екатерина', 'ирина', 'кристина', 'ксения', 'любовь', 'людмила', 'мария', 'марина', 'наталья', 'наталия',
            'ольга', 'светлана', 'софья', 'софия', 'татьяна', 'юлия', 'яна'
        }
        stop_words = {
            'зовут', 'меня', 'я', 'имя', 'это', 'тот', 'эта', 'этот', 'та', 'то', 'был', 'была', 'было', 'были',
            'имел', 'имела', 'имело', 'имели', 'хотел', 'хотела', 'хотело', 'хотели', 'сказал', 'сказала', 'сказало',
            'сказали', 'думал', 'думала', 'думало', 'думали', 'знал', 'знала', 'знало', 'знали', 'видел', 'видела',
            'видело', 'видели', 'делал', 'делала', 'делало', 'делали', 'мог', 'могла', 'могло', 'могли', 'могу',
            'можешь', 'может', 'можем', 'можете', 'могут', 'буду', 'будешь', 'будет', 'будем', 'будете', 'будут',
            'есть', 'быть', 'стал', 'стала', 'стало', 'стали', 'стану', 'станешь', 'станет', 'станем', 'станете',
            'станут', 'вопрос', 'ответ', 'про', 'для', 'что', 'как', 'где', 'когда', 'кто', 'какой', 'какая', 'какое',
            'какие', 'этот', 'эта', 'это', 'эти', 'тот', 'та', 'то', 'те', 'мой', 'моя', 'мое', 'мои', 'твой', 'твоя',
            'твое', 'твои', 'наш', 'наша', 'наше', 'наши', 'ваш', 'ваша', 'ваше', 'ваши'
        }
        for msg in self.conversations[conversation_id]:
            if msg['type'] == 'user':
                content = msg['content'].lower()
                patterns = [
                    r'меня\s+зовут\s+([А-ЯЁа-яё]{3,20})(?:\s|$|[.,!?])',
                    r'меня\s+зовут([А-ЯЁа-яё]{3,20})(?:\s|$|[.,!?])',
                    r'зовут\s+([А-ЯЁа-яё]{3,20})(?:\s|$|[.,!?])',
                    r'имя\s+([А-ЯЁа-яё]{3,20})(?:\s|$|[.,!?])',
                    r'меня\s+([А-ЯЁа-яё]{3,20})\s+зовут',
                ]
                for pattern in patterns:
                    match = re.search(pattern, content)
                    if match:
                        name = match.group(1).capitalize()
                        name_lower = name.lower()
                        if name_lower in stop_words:
                            continue
                        if not name[0].isupper() or not name.isalpha():
                            continue
                        if not (3 <= len(name) <= 20):
                            continue
                        verb_endings = ['ал', 'ала', 'ало', 'али', 'ел', 'ела', 'ело', 'ели', 'ил', 'ила', 'ило', 'или',
                                       'ул', 'ула', 'уло', 'ули', 'ся', 'сь', 'ет', 'ут', 'ат', 'ит', 'ят']
                        if name_lower not in valid_names:
                            if any(name_lower.endswith(ending) for ending in verb_endings):
                                self.logger.debug(f"Пропущено '{name}' - похоже на глагол")
                                continue
                            if not any(vowel in name_lower for vowel in 'аеёиоуыэюя'):
                                continue
                        user_name = name
                        self.logger.info(f"✅ Извлечено имя пользователя из сообщения: '{msg['content'][:50]}...' -> '{user_name}'")
                        break
                if user_name:
                    break
        context_parts = []
        if user_name:
            context_parts.append(f"ВАЖНО: Имя пользователя - {user_name}. Обращайся к нему по имени.")
        for msg in self.conversations[conversation_id][-10:]:
            if msg['type'] == 'user':
                context_parts.append(f"Пользователь: {msg['content']}")
            else:
                context_parts.append(f"Ассистент: {msg['content']}")
        return "\n".join(context_parts)
    def search_documents(self, query: str) -> List[Dict[str, Any]]:

        start_time = time.time()
        try:
            documents = proper_retriever.search(query)
            self.logger.info(f"Найдено {len(documents)} документов для запроса")
            perf_monitor.record_request('search_documents', time.time() - start_time, True)
            return documents
        except Exception as e:
            self.logger.error(f"Ошибка поиска документов: {e}")
            perf_monitor.record_request('search_documents', time.time() - start_time, False)
            try:
                from chroma_client import chroma_client
                documents = chroma_client.search_documents([query], SEARCH_K)
                return documents
            except Exception as fallback_error:
                self.logger.error(f"Fallback поиск также не удался: {fallback_error}")
                return []
    def generate_response(self, question: str, context: str, conversation_context: str) -> str:

        start_time = time.time()
        try:
            sanitized_conversation_context = sanitize_prompt_section(conversation_context)
            sanitized_context = sanitize_prompt_section(context)
            sanitized_question = sanitize_prompt_section(question)
            prompt = CUSTOMER_PROMPT
            prompt = prompt.replace('{customer_notes}', CUSTOMER_NOTES)
            prompt = prompt.replace('{conversation_context}', sanitized_conversation_context)
            prompt = prompt.replace('{context}', sanitized_context)
            prompt = prompt.replace('{question}', sanitized_question)
            request_payload = {
                "model": MODEL_NAME,
                "prompt": prompt,
                "stream": False,
                "keep_alive": KEEP_ALIVE,
                "options": {
                    "temperature": TEMPERATURE,
                    "num_ctx": NUM_CTX,
                    "num_predict": NUM_PREDICT,
                    "repeat_penalty": REPEAT_PENALTY,
                    "stop": STOP_WORDS
                }
            }
            if THINK_LEVEL:
                request_payload["think"] = THINK_LEVEL
            response = ollama_session.post(
                f"{OLLAMA_BASE_URL_FIRST}/api/generate",
                json=request_payload,
                timeout=TIMEOUT
            )
            if response.status_code != 200:
                self.logger.error(f"Ошибка генерации ответа: {response.status_code}")
                return "Извините, произошла ошибка при генерации ответа."
            result = response.json()
            perf_monitor.record_request('generate_response', time.time() - start_time, True)
            return result.get('response', 'Извините, не удалось сгенерировать ответ.')
        except Exception as e:
            self.logger.error(f"Ошибка генерации ответа: {e}")
            perf_monitor.record_request('generate_response', time.time() - start_time, False)
            return "Извините, произошла ошибка при генерации ответа."
    def send_to_analytics(self, conversation_id: str, message_type: str, content: str,
                         llm_generation_time: Optional[float] = None):

        try:
            analytics_data = {
                "type": message_type,
                "content": content,
                "metadata": {
                    "conversation_id": conversation_id,
                    "timestamp": datetime.now().isoformat()
                },
                "llm_generation_time": llm_generation_time
            }
            requests.post(
                f"{ANALYTICS_BASE_URL}/api/conversations/{conversation_id}/messages",
                json=analytics_data,
                timeout=10
            )
        except Exception as e:
            self.logger.error(f"Ошибка отправки в аналитику: {e}")
def load_customer_notes():

    global CUSTOMER_NOTES
    try:
        env = os.getenv('ENVIRONMENT', 'development')
        if env == 'development':
            notes_file = '/opt/autogen/development/configs/customer_notes.txt'
        else:
            notes_file = '/opt/autogen/production/configs/customer_notes.txt'
        if os.path.exists(notes_file):
            with open(notes_file, 'r', encoding='utf-8') as f:
                CUSTOMER_NOTES = f.read().strip()
            logger.info(f"Замечания заказчика загружены из {notes_file}")
        else:
            CUSTOMER_NOTES = ""
            logger.info(f"Файл замечаний не найден: {notes_file}, используется пустая строка")
    except Exception as e:
        logger.error(f"Ошибка загрузки замечаний заказчика: {e}")
        CUSTOMER_NOTES = ""
def load_system_prompt():

    global CUSTOMER_PROMPT
    try:
        system_prompt_path = os.getenv('SYSTEM_PROMPT_PATH')
        if system_prompt_path and os.path.exists(system_prompt_path):
            prompt_file = system_prompt_path
            logger.info(f"Системный промпт загружен из переменной окружения SYSTEM_PROMPT_PATH: {prompt_file}")
        else:
            env = os.getenv('ENVIRONMENT', 'development')
            if env == 'development':
                prompt_file = '/opt/autogen/development/configs/system_prompt.txt'
            else:
                prompt_file = '/opt/autogen/production/configs/system_prompt.txt'
        if os.path.exists(prompt_file):
            with open(prompt_file, 'r', encoding='utf-8') as f:
                CUSTOMER_PROMPT = f.read()
            logger.info(f"Системный промпт загружен из {prompt_file}")
        else:
            logger.warning(f"Файл промпта не найден: {prompt_file}")
            CUSTOMER_PROMPT = """Ты - интеллектуальный RAG-чатбот. Отвечай на русском языке, ставь пробелы между словами и числами."""
    except Exception as e:
        logger.error(f"Ошибка загрузки системного промпта: {e}")
        CUSTOMER_PROMPT = """Ты - интеллектуальный RAG-чатбот. Отвечай на русском языке, ставь пробелы между словами и числами."""
def load_settings():

    global MODEL_NAME, EMBEDDING_MODEL, TEMPERATURE, NUM_CTX, SEARCH_K, TIMEOUT
    global OLLAMA_BASE_URL, CHROMADB_BASE_URL, ANALYTICS_BASE_URL
    global KEEP_ALIVE, THINK_LEVEL
    try:
        MODEL_NAME = os.getenv('MODEL_NAME', MODEL_NAME)
        EMBEDDING_MODEL = os.getenv('EMBEDDING_MODEL', EMBEDDING_MODEL)
        TEMPERATURE = float(os.getenv('TEMPERATURE', TEMPERATURE))
        NUM_CTX = int(os.getenv('NUM_CTX', NUM_CTX))
        SEARCH_K = int(os.getenv('SEARCH_K', SEARCH_K))
        TIMEOUT = int(os.getenv('TIMEOUT', TIMEOUT))
        OLLAMA_BASE_URL = os.getenv('OLLAMA_BASE_URL', OLLAMA_BASE_URL)
        CHROMADB_BASE_URL = os.getenv('CHROMADB_BASE_URL', CHROMADB_BASE_URL)
        ANALYTICS_BASE_URL = os.getenv('ANALYTICS_BASE_URL', ANALYTICS_BASE_URL)
        keep_alive_env = os.getenv('KEEP_ALIVE')
        if keep_alive_env:
            KEEP_ALIVE = parse_keep_alive(keep_alive_env)
        think_level_env = os.getenv('THINK_LEVEL')
        if think_level_env:
            if think_level_env.lower() in ('true', '1', 'yes'):
                THINK_LEVEL = True
            elif think_level_env.lower() in ('false', '0', 'no', ''):
                THINK_LEVEL = False
            else:
                THINK_LEVEL = think_level_env
        max_concurrent = os.getenv('MAX_CONCURRENT')
        if max_concurrent:
            os.environ['MAX_CONCURRENT'] = str(max_concurrent)
            logger.info(f"MAX_CONCURRENT загружен из окружения: {max_concurrent}")
        def normalize_url(url):
            if not url:
                return url
            url = url.replace('http:/', 'http://', 1)
            url = url.replace('http:///', 'http://')
            return url
        OLLAMA_BASE_URL = normalize_url(OLLAMA_BASE_URL)
        CHROMADB_BASE_URL = normalize_url(CHROMADB_BASE_URL)
        ANALYTICS_BASE_URL = normalize_url(ANALYTICS_BASE_URL)
        reranker_service_urls = os.getenv('RERANKER_SERVICE_URLS')
        if reranker_service_urls:
            os.environ['RERANKER_SERVICE_URLS'] = reranker_service_urls
            logger.info(f"✅ RERANKER_SERVICE_URLS установлен из окружения: {reranker_service_urls}")
            try:
                import reranker_client as rc_module
                rc_module.reranker_client.reranker_urls = [url.strip() for url in reranker_service_urls.split(',') if url.strip()]
                rc_module.reranker_client.current_url_index = 0
                logger.info(f"✅ Reranker Client обновлен из окружения: {rc_module.reranker_client.reranker_urls}")
            except Exception as e:
                logger.debug(f"Reranker client еще не загружен, будет инициализирован с правильными URL при первом импорте: {e}")
        logger.info("Настройки загружены из переменных окружения")
    except Exception as e:
        logger.error(f"Ошибка загрузки настроек: {e}")
load_settings()
load_system_prompt()
load_customer_notes()
from proper_retriever import proper_retriever
rag_api = RAGChatAPI()
@app.route('/')
def index():

    return render_template('index.html',
                         analytics_domain=os.getenv('ANALYTICS_DOMAIN'))
@app.route('/widget')
def widget():

    return render_template('widget.html',
                         widget_domain=os.getenv('WIDGET_DOMAIN'),
                         api_domain=os.getenv('API_DOMAIN'))
@app.route('/widget-demo')
def widget_demo():

    return render_template('widget-demo.html',
                         widget_domain=os.getenv('WIDGET_DOMAIN'),
                         api_domain=os.getenv('API_DOMAIN'))
@app.route('/embed')
def embed():

    return send_from_directory('static', 'embed.html')
@app.route('/admin')
def admin():

    reranker_service_urls = os.getenv('RERANKER_SERVICE_URLS', '')
    widget_domain = os.getenv('WIDGET_DOMAIN')
    api_domain = os.getenv('API_DOMAIN')
    analytics_domain = os.getenv('ANALYTICS_DOMAIN')
    analytics_host = os.getenv('ANALYTICS_HOST', 'dev-analytics')
    analytics_port = os.getenv('ANALYTICS_PORT', '9005')
    try:
        env = os.getenv('ENVIRONMENT', 'development')
        settings_file = '/opt/autogen/settings.dev.json' if env == 'development' else '/opt/autogen/settings.prod.json'
        if not os.path.exists(settings_file):
            settings_file = '/opt/autogen/settings.json'
        if os.path.exists(settings_file):
            with open(settings_file, 'r', encoding='utf-8') as f:
                settings = json.load(f)
                reranker_service_urls = settings.get('reranker_service_urls', reranker_service_urls)
                widget_domain = settings.get('widget_domain', widget_domain)
                api_domain = settings.get('api_domain', api_domain)
                analytics_domain = settings.get('analytics_domain', analytics_domain)
                analytics_host = settings.get('analytics_host', analytics_host)
                analytics_port = settings.get('analytics_port', analytics_port)
    except:
        pass
    return render_template('admin.html',
                         ollama_url=OLLAMA_BASE_URL,
                         chromadb_url=CHROMADB_BASE_URL,
                         analytics_url=ANALYTICS_BASE_URL,
                         reranker_service_urls=reranker_service_urls,
                         ollama_host=os.getenv('OLLAMA_HOST', ''),
                         ollama_port=os.getenv('OLLAMA_PORT', ''),
                         chromadb_host=os.getenv('CHROMADB_HOST', ''),
                         chromadb_port=os.getenv('CHROMADB_PORT', ''),
                         analytics_host=analytics_host,
                         analytics_port=analytics_port,
                         widget_domain=widget_domain,
                         api_domain=api_domain,
                         analytics_domain=analytics_domain)
@app.route('/api/chat/stream', methods=['POST'])
def chat_stream():

    try:
        data = request.get_json()
        question = data.get('message', data.get('question', '')).strip()
        conversation_id = data.get('conversation_id')
        temperature = data.get('temperature', TEMPERATURE)
        search_k = data.get('search_k', SEARCH_K)
        if not question:
            return jsonify({'error': 'Вопрос не может быть пустым'}), 400
        if not conversation_id:
            session_data = {
                'ip_address': request.remote_addr,
                'user_agent': request.headers.get('User-Agent', '')
            }
            conversation_id = rag_api.generate_conversation_id(session_data)
        if conversation_id not in rag_api.conversations:
            rag_api.conversations[conversation_id] = []
        rag_api.conversations[conversation_id].append({
            'type': 'user',
            'content': question,
            'timestamp': datetime.now().isoformat()
        })
        conversation_context_raw = rag_api.get_conversation_context(conversation_id)
        msg_count = len(rag_api.conversations.get(conversation_id, []))
        context_chars = len(conversation_context_raw)
        context_tokens_estimate = context_chars // 4
        if "ВАЖНО: Имя пользователя" in conversation_context_raw:
            name_line = [line for line in conversation_context_raw.split('\n') if 'Имя пользователя' in line]
            if name_line:
                logger.info(f"👤 {name_line[0]}")
        logger.info(f"📜 История диалога: {msg_count} сообщений, {context_chars} символов (~{context_tokens_estimate} токенов)")
        if msg_count > 0:
            logger.info(f"   Последние сообщения: {[m['type'] + ':' + m['content'][:30] + '...' for m in rag_api.conversations.get(conversation_id, [])[-3:]]}")
        logger.debug(f"🔍 Начало conversation_context: {conversation_context_raw[:200]}...")
        system_prompt_size = len(CUSTOMER_PROMPT.replace('{customer_notes}', CUSTOMER_NOTES).replace('{conversation_context}', '').replace('{context}', '').replace('{question}', ''))
        max_total_tokens = int(NUM_CTX * PROMPT_CONTEXT_GUARD_RATIO)
        max_total_chars = max_total_tokens * 4
        reserved_chars = system_prompt_size + len(question) + 2000
        available_chars = max_total_chars - reserved_chars
        max_conv_chars = int(available_chars * 0.3)
        max_docs_chars = int(available_chars * 0.7)
        logger.debug(f"📊 Лимиты контекста: всего={max_total_chars} chars (~{max_total_tokens} токенов), system_prompt={system_prompt_size} chars, доступно={available_chars} chars")
        logger.debug(f"   conversation_context: до {max_conv_chars} chars (~{max_conv_chars//4} токенов)")
        logger.debug(f"   documents_context: до {max_docs_chars} chars (~{max_docs_chars//4} токенов)")
        conversation_context = enforce_context_limit(conversation_context_raw, "conversation_context", max_conv_chars)
        sanitized_conversation_context = sanitize_prompt_section(conversation_context)
        logger.debug(f"🔍 conversation_context после enforce_limit: {len(conversation_context)} символов (~{len(conversation_context)//4} токенов)")
        logger.debug(f"🔍 sanitized_conversation_context: {len(sanitized_conversation_context)} символов")
        start_time = time.time()
        documents = rag_api.search_documents(question)
        search_time = time.time() - start_time
        context_parts = []
        logger.info(f"DEBUG_SEARCH: Found {len(documents)} documents for query: {question}")
        if logger.isEnabledFor(logging.DEBUG):
            query_lower = question.lower()
            query_keywords = set(query_lower.split())
            stop_words = {'в', 'на', 'и', 'а', 'но', 'по', 'с', 'у', 'к', 'о', 'это', 'как', 'его', 'что', 'для', 'когда', 'где', 'кто', 'какой'}
            query_keywords = {w for w in query_keywords if len(w) > 2 and w not in stop_words}
            logger.debug(f"DEBUG_SEARCH: Query keywords: {query_keywords}")
            for i, doc in enumerate(documents[:5], 1):
                has_20_april = '20 апреля' in doc.get('content', '')
                has_1_march = '1 марта' in doc.get('content', '')
                dist = doc.get('distance', 0)
                src = doc.get('metadata', {}).get('source', 'Unknown')
                logger.debug(f"DEBUG_SEARCH: {i}. dist={dist:.4f} | 20apr:{has_20_april} | 1mar:{has_1_march} | {src[:50]}")
        relevant_docs = documents
        if not relevant_docs:
            logger.info(f"Не найдено релевантных документов для запроса: {question}")
            context = "Документы не найдены или не релевантны запросу."
        else:
            logger.info(f"Отобрано {len(relevant_docs)} релевантных документов из {len(documents)} найденных")
            for doc in relevant_docs:
                context_parts.append(f"Источник: {doc.get('metadata', {}).get('source', 'Неизвестно')}\n{doc['content']}")
            context = "\n\n".join(context_parts)
        context = enforce_context_limit(context, "documents_context", max_docs_chars)
        sanitized_context = sanitize_prompt_section(context)
        sanitized_question = sanitize_prompt_section(question)
        context_docs_chars = len(context)
        context_docs_tokens = context_docs_chars // 4
        logger.info(f"📚 Контекст документов: {context_docs_chars} символов (~{context_docs_tokens} токенов) из {len(documents)} документов")
        stream_start_time = time.time()
        generation_statuses = [
            (0, "Генерация ответа..."),
            (10, "Проверка точности информации..."),
            (30, "Запрос требует больше времени, пожалуйста подождите..."),
            (60, "Обработка сложного запроса, это может занять некоторое время...")
        ]
        def make_status_event(status: str, message: str) -> str:
            return f"data: {json.dumps({'type': 'status', 'status': status, 'message': message})}\n\n"
        def generate_stream():
            yield f"data: {json.dumps({'type': 'start', 'conversation_id': conversation_id})}\n\n"
            yield make_status_event('searching', 'Поиск информации в базе знаний...')
            yield make_status_event('processing', "Нашла релевантные материалы. Извлекаю ключевые факты и проверяю даты из подтверждённых источников...")
            generation_start = time.time()
            yield make_status_event('generating', generation_statuses[0][1])
            next_status_idx = 1
            prompt = CUSTOMER_PROMPT
            prompt = prompt.replace('{customer_notes}', CUSTOMER_NOTES)
            prompt = prompt.replace('{conversation_context}', sanitized_conversation_context)
            prompt = prompt.replace('{context}', sanitized_context)
            prompt = prompt.replace('{question}', sanitized_question)
            prompt_chars = len(prompt)
            prompt_tokens = prompt_chars // 4
            max_allowed_tokens = int(NUM_CTX * PROMPT_CONTEXT_GUARD_RATIO)
            if prompt_tokens > max_allowed_tokens:
                logger.error(f"❌ ПРОМПТ ПРЕВЫСИЛ ЛИМИТ: {prompt_tokens} токенов > {max_allowed_tokens} токенов (превышение на {prompt_tokens - max_allowed_tokens} токенов)")
                logger.error(f"   Это может привести к ошибкам генерации или обрезке контекста моделью!")
            else:
                logger.info(f"📤 Полный промпт: {prompt_chars} символов (~{prompt_tokens} токенов)")
            conv_ctx_size = len(sanitized_conversation_context)
            docs_ctx_size = len(sanitized_context)
            question_size = len(sanitized_question)
            system_size = prompt_chars - conv_ctx_size - docs_ctx_size - question_size
            logger.info(f"📈 Использование контекста: {prompt_tokens}/{NUM_CTX} токенов ({prompt_tokens*100//NUM_CTX if NUM_CTX > 0 else 0}%)")
            logger.debug(f"   📋 Структура промпта:")
            logger.debug(f"      - System prompt: ~{system_size} chars (~{system_size//4} токенов)")
            logger.debug(f"      - Conversation context: {conv_ctx_size} chars (~{conv_ctx_size//4} токенов)")
            logger.debug(f"      - Documents context: {docs_ctx_size} chars (~{docs_ctx_size//4} токенов)")
            logger.debug(f"      - Question: {question_size} chars (~{question_size//4} токенов)")
            logger.debug(f"🔍 Начало промпта (первые 500 символов): {prompt[:500]}...")
            if "ВАЖНО: Имя пользователя" in prompt:
                name_line = [line for line in prompt.split('\n') if 'Имя пользователя' in line]
                if name_line:
                    logger.info(f"✅ {name_line[0]}")
            else:
                logger.warning("⚠️  Информация об имени пользователя НЕ найдена в промпте")
            if THINK_LEVEL and "размышления" not in prompt.lower() and "thinking" not in prompt.lower():
                thinking_instruction = "\n\n⚠️ ВАЖНО: Все размышления (thinking) должны быть на русском языке и краткими."
                prompt = thinking_instruction + "\n\n" + prompt
            prompt_chars = len(prompt)
            prompt_tokens = prompt_chars // 4
            logger.info(f"📊 RAG контекст: {len(documents)} документов, {context_docs_chars} символов (~{context_docs_tokens} токенов)")
            logger.info(f"📤 Полный промпт: {prompt_chars} символов (~{prompt_tokens} токенов)")
            logger.info(f"📈 Использование контекста: {prompt_tokens}/{NUM_CTX} токенов ({prompt_tokens*100//NUM_CTX}%)")
            logger.info(f"🔧 Модель: {MODEL_NAME}, температура: {temperature}")
            try:
                request_payload = {
                    "model": MODEL_NAME,
                    "prompt": prompt,
                    "stream": True,
                    "keep_alive": KEEP_ALIVE,
                    "options": {
                        "temperature": temperature,
                        "num_ctx": NUM_CTX,
                        "num_predict": NUM_PREDICT,
                        "repeat_penalty": REPEAT_PENALTY,
                        "stop": STOP_WORDS
                    }
                }
                if THINK_LEVEL:
                    request_payload["think"] = THINK_LEVEL
                ollama_url = OLLAMA_BASE_URL_FIRST.rstrip('/') + '/api/generate'
                ollama_response = ollama_session.post(
                    ollama_url,
                    json=request_payload,
                    timeout=TIMEOUT,
                    stream=True
                )
                if ollama_response.status_code != 200:
                    yield make_status_event('error', f"Ошибка генерации ответа (код {ollama_response.status_code}). Пожалуйста, попробуйте позже.")
                    return
                response_text = ""
                token_count = 0
                generation_loop_start = time.time()
                last_event_time = generation_loop_start
                first_token_timeout = 30
                max_idle_time = 60
                received_first_token = False
                stream_completed = False
                last_data = None
                logger.info(f"📥 Начинаю получать streaming ответ от Ollama...")
                for line in ollama_response.iter_lines():
                    if not line:
                        continue
                    try:
                        data = json.loads(line.decode('utf-8'))
                        last_data = data
                    except json.JSONDecodeError as e:
                        logger.warning(f"Ошибка парсинга JSON от Ollama: {e}, строка: {line[:100]}")
                        continue
                    if 'error' in data:
                        error_msg = data.get('error', 'Неизвестная ошибка от Ollama')
                        logger.error(f"Ошибка от Ollama: {error_msg}")
                        yield make_status_event('error', f'Ошибка генерации: {error_msg}')
                        return
                    response_token = None
                    thinking_token = None
                    if 'response' in data and data['response']:
                        response_token = data['response']
                    if 'thinking' in data and data['thinking']:
                        thinking_token = data['thinking']
                    if thinking_token:
                        yield f"data: {json.dumps({'type': 'thinking', 'content': thinking_token})}\n\n"
                        sys.stdout.flush()
                    if response_token:
                        if not received_first_token:
                            received_first_token = True
                            elapsed = time.time() - generation_loop_start
                            if elapsed > first_token_timeout:
                                logger.warning(f"Первый токен получен слишком поздно: {elapsed:.2f}с")
                        response_text += response_token
                        current_time = time.time()
                        token_count += 1
                        last_event_time = current_time
                        elapsed = current_time - stream_start_time
                        while next_status_idx < len(generation_statuses) and elapsed >= generation_statuses[next_status_idx][0]:
                            yield make_status_event('generating', generation_statuses[next_status_idx][1])
                            next_status_idx += 1
                        yield f"data: {json.dumps({'type': 'token', 'content': response_token})}\n\n"
                        sys.stdout.flush()
                    else:
                        current_time = time.time()
                        idle_time = current_time - last_event_time
                        if received_first_token and idle_time > max_idle_time:
                            logger.warning(f"Превышен таймаут простоя: {idle_time:.2f}с без токенов")
                            break
                    if data.get('done', False):
                        stream_completed = True
                        done_reason = data.get('done_reason', 'unknown')
                        logger.info(f"Генерация завершена. Всего токенов: {token_count}, длина ответа: {len(response_text)} символов, done_reason: {done_reason}")
                        if not response_text or not response_text.strip():
                            logger.warning(f"⚠️ Стрим завершен (done=True), но ответ пустой! done_reason: {done_reason}, токенов: {token_count}")
                        break
                if not stream_completed:
                    elapsed = time.time() - generation_loop_start
                    logger.warning(f"Стрим от Ollama завершился без флага done. Время: {elapsed:.2f}с, токенов: {token_count}, последний data: {last_data}")
                if not received_first_token:
                    elapsed = time.time() - generation_loop_start
                    done_reason = last_data.get('done_reason', 'N/A') if last_data else 'N/A'
                    logger.error(f"КРИТИЧЕСКАЯ ОШИБКА: Не получен ни один токен от Ollama за {elapsed:.2f}с. Модель: {MODEL_NAME}, stream_completed: {stream_completed}, done_reason: {done_reason}")
                    logger.error(f"  Последний data: {last_data}")
                    if done_reason == 'length':
                        logger.error(f"  ⚠️ Модель достигла лимита длины (num_predict={NUM_PREDICT}), но не успела выдать ответ. Возможно, нужно увеличить num_predict или уменьшить промпт.")
                        yield make_status_event('error', 'Модель достигла лимита длины ответа. Попробуйте переформулировать вопрос короче или повторить запрос.')
                    else:
                        yield make_status_event('error', 'Модель не вернула ответ. Попробуйте переформулировать вопрос или повторить запрос.')
                    return
                if not response_text or not response_text.strip():
                    elapsed = time.time() - generation_loop_start
                    done_reason = last_data.get('done_reason', 'N/A') if last_data else 'N/A'
                    logger.error(f"КРИТИЧЕСКАЯ ОШИБКА: Ответ от Ollama пустой! Модель: {MODEL_NAME}, токенов получено: {token_count}, stream_completed: {stream_completed}, done_reason: {done_reason}, время: {elapsed:.2f}с")
                    logger.error(f"  Последний data от Ollama: {last_data}")
                    if done_reason == 'length':
                        logger.error(f"  ⚠️ Модель достигла лимита длины (num_predict={NUM_PREDICT}), но ответ пустой. Возможно, нужно увеличить num_predict или уменьшить промпт.")
                        yield make_status_event('error', 'Модель достигла лимита длины ответа. Попробуйте переформулировать вопрос короче или повторить запрос.')
                    else:
                        yield make_status_event('error', 'Модель вернула пустой ответ. Попробуйте переформулировать вопрос или повторить запрос.')
                    return
                if token_count < 5 and response_text.strip():
                    logger.warning(f"Подозрительно короткий ответ: {token_count} токенов, {len(response_text)} символов. Модель: {MODEL_NAME}")
            except requests.exceptions.Timeout as e:
                logger.error(f"Таймаут при запросе к Ollama: {e}")
                yield make_status_event('error', 'Превышено время ожидания ответа. Попробуйте задать вопрос короче или переформулировать запрос.')
                return
            except requests.exceptions.ConnectionError as e:
                logger.error(f"Ошибка подключения к Ollama: {e}")
                yield make_status_event('error', 'Не удалось подключиться к модели. Попробуйте повторить запрос через несколько секунд.')
                return
            except requests.exceptions.RequestException as e:
                logger.error(f"Ошибка HTTP запроса к Ollama: {e}")
                yield make_status_event('error', 'Ошибка при обращении к модели. Попробуйте повторить запрос.')
                return
            except Exception as e:
                logger.error(f"Ошибка при генерации ответа: {e}", exc_info=True)
                yield make_status_event('error', 'Произошла ошибка при генерации ответа. Попробуйте повторить запрос.')
                return
            generation_time = time.time() - generation_start
            if not response_text or not response_text.strip():
                elapsed = time.time() - generation_start
                logger.error(f"КРИТИЧЕСКАЯ ОШИБКА: Ответ пустой после завершения генерации! Модель: {MODEL_NAME}, токенов: {token_count}, время: {elapsed:.2f}с")
                logger.error(f"  response_text длина: {len(response_text)}, strip длина: {len(response_text.strip()) if response_text else 0}")
                yield make_status_event('error', 'Модель вернула пустой ответ. Попробуйте переформулировать вопрос.')
                return
            logger.info(f"✅ Успешная генерация: {token_count} токенов, {len(response_text)} символов, время: {generation_time:.2f}с")
            rag_api.conversations[conversation_id].append({
                'type': 'assistant',
                'content': response_text,
                'timestamp': datetime.now().isoformat()
            })
            rag_api.send_to_analytics(conversation_id, 'user', question)
            rag_api.send_to_analytics(conversation_id, 'assistant', response_text, generation_time)
            yield make_status_event('complete', 'Ответ готов.')
            yield f"data: {json.dumps({'type': 'end', 'metadata': {'search_time': search_time, 'generation_time': generation_time, 'documents_found': len(documents)}})}\n\n"
        return Response(generate_stream(), mimetype='text/event-stream')
    except Exception as e:
        logger.error(f"Ошибка в chat_stream: {e}")
        return jsonify({'error': 'Внутренняя ошибка сервера'}), 500
@app.route('/api/status')
def status():

    try:
        ollama_status = "ok"
        try:
            response = requests.get(f"{OLLAMA_BASE_URL_FIRST}/api/tags", timeout=5)
            if response.status_code != 200:
                ollama_status = "error"
        except:
            ollama_status = "error"
        chromadb_status = "ok"
        try:
            from chroma_client import chroma_client
            chroma_client.client.list_collections()
        except:
            chromadb_status = "error"
        analytics_status = "ok"
        try:
            response = requests.get(f"{ANALYTICS_BASE_URL}/api/stats", timeout=5)
            if response.status_code != 200:
                analytics_status = "error"
        except:
            analytics_status = "error"
        return jsonify({
            'status': 'ok',
            'services': {
                'ollama': ollama_status,
                'chromadb': chromadb_status,
                'analytics': analytics_status
            },
            'config': {
                'model': MODEL_NAME,
                'embedding_model': EMBEDDING_MODEL,
                'temperature': TEMPERATURE,
                'num_ctx': NUM_CTX,
                'search_k': SEARCH_K,
                'timeout': TIMEOUT,
                'keep_alive': KEEP_ALIVE,
                'think_level': THINK_LEVEL
            }
        })
    except Exception as e:
        logger.error(f"Ошибка в status: {e}")
        return jsonify({'error': 'Ошибка проверки статуса'}), 500
@app.route('/api/widget/config')
def widget_config():

    return jsonify({
        'temperature': TEMPERATURE,
        'search_k': SEARCH_K,
        'model': MODEL_NAME,
        'embedding_model': EMBEDDING_MODEL,
        'num_ctx': NUM_CTX,
        'timeout': TIMEOUT,
        'keep_alive': KEEP_ALIVE,
        'think_level': THINK_LEVEL
    })
@app.route('/api/settings')
def settings():

    reranker_service_urls = os.getenv('RERANKER_SERVICE_URLS', '')
    max_concurrent = int(os.getenv('MAX_CONCURRENT', '10'))
    widget_domain = os.getenv('WIDGET_DOMAIN')
    api_domain = os.getenv('API_DOMAIN')
    analytics_domain = os.getenv('ANALYTICS_DOMAIN')
    analytics_host = os.getenv('ANALYTICS_HOST')
    analytics_port = os.getenv('ANALYTICS_PORT')
    try:
        env = os.getenv('ENVIRONMENT', 'development')
        settings_file = '/opt/autogen/settings.dev.json' if env == 'development' else '/opt/autogen/settings.prod.json'
        if not os.path.exists(settings_file):
            settings_file = '/opt/autogen/settings.json'
        if os.path.exists(settings_file):
            with open(settings_file, 'r', encoding='utf-8') as f:
                settings_data = json.load(f)
                reranker_service_urls = settings_data.get('reranker_service_urls', reranker_service_urls)
                max_concurrent = settings_data.get('max_concurrent', max_concurrent)
                widget_domain = settings_data.get('widget_domain', widget_domain)
                api_domain = settings_data.get('api_domain', api_domain)
                analytics_domain = settings_data.get('analytics_domain', analytics_domain)
                analytics_host = settings_data.get('analytics_host', analytics_host)
                analytics_port = settings_data.get('analytics_port', analytics_port)
    except:
        pass
    return jsonify({
        'model': MODEL_NAME,
        'embedding_model': EMBEDDING_MODEL,
        'temperature': TEMPERATURE,
        'num_ctx': NUM_CTX,
        'search_k': SEARCH_K,
        'timeout': TIMEOUT,
        'keep_alive': KEEP_ALIVE,
        'think_level': THINK_LEVEL,
        'ollama_url': OLLAMA_BASE_URL,
        'chromadb_url': CHROMADB_BASE_URL,
        'analytics_url': ANALYTICS_BASE_URL,
        'reranker_service_urls': reranker_service_urls,
        'max_concurrent': max_concurrent,
        'widget_domain': widget_domain,
        'api_domain': api_domain,
        'analytics_domain': analytics_domain,
        'analytics_host': analytics_host,
        'analytics_port': analytics_port
    })
@app.route('/api/settings', methods=['POST'])
def update_settings():

    try:
        data = request.get_json()
        global MODEL_NAME, EMBEDDING_MODEL, TEMPERATURE, NUM_CTX, SEARCH_K, TIMEOUT
        global OLLAMA_BASE_URL, CHROMADB_BASE_URL, ANALYTICS_BASE_URL
        global THINK_LEVEL
        if 'model' in data:
            MODEL_NAME = data['model']
        if 'embedding_model' in data:
            EMBEDDING_MODEL = data['embedding_model']
        if 'temperature' in data:
            TEMPERATURE = float(data['temperature'])
        if 'num_ctx' in data:
            NUM_CTX = int(data['num_ctx'])
        if 'search_k' in data:
            SEARCH_K = int(data['search_k'])
        if 'timeout' in data:
            TIMEOUT = int(data['timeout'])
        if 'ollama_url' in data:
            OLLAMA_BASE_URL = data['ollama_url']
        if 'chromadb_url' in data:
            CHROMADB_BASE_URL = data['chromadb_url']
        if 'analytics_url' in data:
            ANALYTICS_BASE_URL = data['analytics_url']
        if 'think_level' in data:
            think_level = data['think_level']
            if isinstance(think_level, bool):
                THINK_LEVEL = think_level
            elif isinstance(think_level, str):
                if think_level.lower() in ('true', '1', 'yes'):
                    THINK_LEVEL = True
                elif think_level.lower() in ('false', '0', 'no', ''):
                    THINK_LEVEL = False
                else:
                    THINK_LEVEL = think_level
            else:
                THINK_LEVEL = think_level
        if 'reranker_service_urls' in data:
            reranker_service_urls = data['reranker_service_urls']
            os.environ['RERANKER_SERVICE_URLS'] = reranker_service_urls
            logger.info(f"RERANKER_SERVICE_URLS обновлен: {reranker_service_urls}")
            try:
                from reranker_client import reranker_client
                reranker_client.reranker_urls = [url.strip() for url in reranker_service_urls.split(',') if url.strip()]
                reranker_client.current_url_index = 0
                logger.info(f"Reranker Client обновлен с новыми URL: {reranker_client.reranker_urls}")
            except Exception as e:
                logger.warning(f"Не удалось обновить reranker_client: {e}")
        if 'system_prompt' in data:
            env = os.getenv('ENVIRONMENT', 'development')
            if env == 'development':
                try:
                    prompt_file = '/opt/autogen/development/configs/system_prompt.txt'
                    with open(prompt_file, 'w', encoding='utf-8') as f:
                        f.write(data['system_prompt'])
                    load_system_prompt()
                    logger.info(f"Системный промпт сохранен в {prompt_file}")
                except Exception as e:
                    logger.error(f"Ошибка сохранения системного промпта: {e}")
                    return jsonify({'error': 'Ошибка сохранения промпта'}), 500
            else:
                logger.warning("Изменение системного промпта разрешено только в dev среде")
                return jsonify({'error': 'Изменение промпта разрешено только в dev среде'}), 403
        if 'customer_notes' in data:
            env = os.getenv('ENVIRONMENT', 'development')
            try:
                if env == 'development':
                    notes_file = '/opt/autogen/development/configs/customer_notes.txt'
                else:
                    notes_file = '/opt/autogen/production/configs/customer_notes.txt'
                with open(notes_file, 'w', encoding='utf-8') as f:
                    f.write(data['customer_notes'])
                load_customer_notes()
                logger.info(f"Замечания заказчика сохранены в {notes_file}")
            except Exception as e:
                logger.error(f"Ошибка сохранения замечаний заказчика: {e}")
                return jsonify({'error': 'Ошибка сохранения замечаний'}), 500
        max_concurrent = None
        if 'max_concurrent' in data:
            max_concurrent = int(data['max_concurrent'])
            os.environ['MAX_CONCURRENT'] = str(max_concurrent)
            logger.info(f"MAX_CONCURRENT установлен: {max_concurrent}")
        reranker_service_urls = data.get('reranker_service_urls') or os.getenv('RERANKER_SERVICE_URLS', '')
        widget_domain = data.get('widget_domain') or os.getenv('WIDGET_DOMAIN')
        api_domain = data.get('api_domain') or os.getenv('API_DOMAIN')
        analytics_domain = data.get('analytics_domain') or os.getenv('ANALYTICS_DOMAIN')
        analytics_host = data.get('analytics_host') or os.getenv('ANALYTICS_HOST')
        analytics_port = data.get('analytics_port') or os.getenv('ANALYTICS_PORT')
        def normalize_url(url):
            if not url:
                return url
            url = url.replace('http:/', 'http://', 1)
            url = url.replace('http:///', 'http://')
            return url
        if widget_domain:
            os.environ['WIDGET_DOMAIN'] = normalize_url(widget_domain)
        if api_domain:
            os.environ['API_DOMAIN'] = normalize_url(api_domain)
        if analytics_domain:
            os.environ['ANALYTICS_DOMAIN'] = normalize_url(analytics_domain)
        if analytics_host:
            os.environ['ANALYTICS_HOST'] = analytics_host
        if analytics_port:
            os.environ['ANALYTICS_PORT'] = str(analytics_port)
        settings_data = {
            'model': MODEL_NAME,
            'embedding_model': EMBEDDING_MODEL,
            'temperature': TEMPERATURE,
            'num_ctx': NUM_CTX,
            'search_k': SEARCH_K,
            'timeout': TIMEOUT,
            'think_level': THINK_LEVEL,
            'ollama_url': normalize_url(OLLAMA_BASE_URL),
            'chromadb_url': normalize_url(CHROMADB_BASE_URL),
            'analytics_url': normalize_url(ANALYTICS_BASE_URL),
            'reranker_service_urls': reranker_service_urls,
            'widget_domain': normalize_url(widget_domain) if widget_domain else None,
            'api_domain': normalize_url(api_domain) if api_domain else None,
            'analytics_domain': normalize_url(analytics_domain) if analytics_domain else None,
            'analytics_host': analytics_host,
            'analytics_port': int(analytics_port) if analytics_port else None,
            'updated_at': datetime.now().isoformat()
        }
        if max_concurrent is not None:
            settings_data['max_concurrent'] = max_concurrent
        env = os.getenv('ENVIRONMENT', 'development')
        settings_file = '/opt/autogen/settings.dev.json' if env == 'development' else '/opt/autogen/settings.prod.json'
        with open(settings_file, 'w', encoding='utf-8') as f:
            json.dump(settings_data, f, indent=2, ensure_ascii=False)
        logger.info(f"Настройки сохранены в {settings_file}: {settings_data}")
        return jsonify({
            'status': 'success',
            'message': 'Настройки сохранены',
            'settings': settings_data
        })
    except Exception as e:
        logger.error(f"Ошибка обновления настроек: {e}")
        return jsonify({'error': 'Ошибка сохранения настроек'}), 500
@app.route('/api/system-prompt', methods=['GET'])
def get_system_prompt():

    try:
        return jsonify({
            'system_prompt': CUSTOMER_PROMPT,
            'environment': os.getenv('ENVIRONMENT', 'development')
        })
    except Exception as e:
        logger.error(f"Ошибка получения системного промпта: {e}")
        return jsonify({'error': 'Ошибка получения промпта'}), 500
@app.route('/api/customer-notes', methods=['GET'])
def get_customer_notes():

    try:
        return jsonify({
            'customer_notes': CUSTOMER_NOTES,
            'environment': os.getenv('ENVIRONMENT', 'development')
        })
    except Exception as e:
        logger.error(f"Ошибка получения замечаний заказчика: {e}")
        return jsonify({'error': 'Ошибка получения замечаний'}), 500
@app.route('/api/admin/stats')
def admin_stats():

    try:
        analytics_response = requests.get(f"{ANALYTICS_BASE_URL}/api/stats", timeout=5)
        analytics_data = analytics_response.json() if analytics_response.status_code == 200 else {}
        services_status = {}
        try:
            ollama_response = requests.get(f"{OLLAMA_BASE_URL}/api/tags", timeout=5)
            services_status['ollama'] = 'ok' if ollama_response.status_code == 200 else 'error'
        except:
            services_status['ollama'] = 'error'
        try:
            from chroma_client import chroma_client
            chroma_client.client.list_collections()
            services_status['chromadb'] = 'ok'
        except:
            services_status['chromadb'] = 'error'
        try:
            analytics_status_response = requests.get(f"{ANALYTICS_BASE_URL}/api/stats", timeout=5)
            services_status['analytics'] = 'ok' if analytics_status_response.status_code == 200 else 'error'
        except:
            services_status['analytics'] = 'error'
        return jsonify({
            'status': 'ok',
            'services': services_status,
            'analytics': analytics_data.get('stats', {}),
            'system': {
                'uptime': 'N/A',
                'version': '2.0.0',
                'widget_version': '1.6.3',
                'last_updated': datetime.now().isoformat()
            }
        })
    except Exception as e:
        logger.error(f"Ошибка получения статистики: {e}")
        return jsonify({'error': 'Ошибка получения статистики'}), 500
@app.route('/analytics')
def proxy_analytics():

    try:
        logger.info(f"[DEV] Проксирование analytics на: {ANALYTICS_BASE_URL}")
        upstream = requests.get(f"{ANALYTICS_BASE_URL}/", timeout=10)
        from flask import make_response
        if upstream.status_code == 200:
            resp = make_response(upstream.text, 200)
        else:
            resp = make_response(f"Analytics недоступен: {upstream.status_code}", upstream.status_code)
        resp.headers['Cache-Control'] = 'no-store, no-cache, must-revalidate, max-age=0'
        resp.headers['Pragma'] = 'no-cache'
        resp.headers['Expires'] = '0'
        resp.headers['Vary'] = 'Accept-Encoding'
        return resp
    except Exception as e:
        logger.error(f"Ошибка проксирования analytics: {e}")
        return f"Ошибка подключения к analytics: {e}", 500
@app.route('/api/stats')
def proxy_stats():

    try:
        upstream = requests.get(f"{ANALYTICS_BASE_URL}/api/stats", timeout=10)
        from flask import make_response
        data = upstream.json() if upstream.status_code == 200 else {'error': 'Analytics API недоступен'}
        resp = make_response(jsonify(data), upstream.status_code)
        resp.headers['Cache-Control'] = 'no-store, no-cache, must-revalidate, max-age=0'
        resp.headers['Pragma'] = 'no-cache'
        resp.headers['Expires'] = '0'
        resp.headers['Vary'] = 'Accept-Encoding'
        return resp
    except Exception as e:
        logger.error(f"Ошибка проксирования stats: {e}")
        return jsonify({'error': 'Ошибка подключения к analytics API'}), 500
@app.route('/api/conversations')
def proxy_conversations():

    try:
        upstream = requests.get(f"{ANALYTICS_BASE_URL}/api/conversations", timeout=10)
        from flask import make_response
        data = upstream.json() if upstream.status_code == 200 else {'error': 'Analytics API недоступен'}
        resp = make_response(jsonify(data), upstream.status_code)
        resp.headers['Cache-Control'] = 'no-store, no-cache, must-revalidate, max-age=0'
        resp.headers['Pragma'] = 'no-cache'
        resp.headers['Expires'] = '0'
        resp.headers['Vary'] = 'Accept-Encoding'
        return resp
    except Exception as e:
        logger.error(f"Ошибка проксирования conversations: {e}")
        return jsonify({'error': 'Ошибка подключения к analytics API'}), 500
@app.route('/api/conversations/<conversation_id>')
def proxy_conversation_details(conversation_id):

    try:
        upstream = requests.get(f"{ANALYTICS_BASE_URL}/api/conversations/{conversation_id}", timeout=10)
        from flask import make_response
        data = upstream.json() if upstream.status_code == 200 else {'error': 'Analytics API недоступен'}
        resp = make_response(jsonify(data), upstream.status_code)
        resp.headers['Cache-Control'] = 'no-store, no-cache, must-revalidate, max-age=0'
        resp.headers['Pragma'] = 'no-cache'
        resp.headers['Expires'] = '0'
        resp.headers['Vary'] = 'Accept-Encoding'
        return resp
    except Exception as e:
        logger.error(f"Ошибка проксирования conversation details: {e}")
        return jsonify({'error': 'Ошибка подключения к analytics API'}), 500
@app.route('/api/analytics/<path:path>')
def proxy_analytics_api(path):

    try:
        url = f"{ANALYTICS_BASE_URL}/api/{path}"
        response = requests.get(url, timeout=10)
        if response.status_code == 200:
            return jsonify(response.json())
        else:
            return jsonify({'error': 'Analytics API недоступен'}), response.status_code
    except Exception as e:
        logger.error(f"Ошибка проксирования analytics API: {e}")
        return jsonify({'error': 'Ошибка подключения к analytics API'}), 500
@app.route('/api/admin/test', methods=['POST'])
def admin_test():

    import time
    def measure(label: str, func):
        start = time.time()
        result = {
            'status': 'error',
            'latency': None,
            'message': '',
        }
        try:
            ok, message = func()
            result['status'] = 'ok' if ok else 'error'
            result['message'] = message
        except Exception as exc:
            logger.error(f"Admin test: {label} failed - {exc}")
            result['status'] = 'error'
            result['message'] = str(exc)
        finally:
            result['latency'] = round(time.time() - start, 3)
        return result
    def test_ollama():
        test_payload = {
            "model": MODEL_NAME,
            "prompt": "Системный тест работоспособности",
            "stream": False,
            "keep_alive": KEEP_ALIVE,
            "options": {
                "num_predict": 32,
                "num_ctx": min(NUM_CTX, 1024),
                "repeat_penalty": REPEAT_PENALTY
            }
        }
        if THINK_LEVEL:
            test_payload["think"] = THINK_LEVEL
        response = requests.post(
            f"{OLLAMA_BASE_URL_FIRST}/api/generate",
            json=test_payload,
            timeout=300
        )
        if response.status_code == 200:
            return True, "Ответ получен от Ollama"
        return False, f"Ollama вернула код {response.status_code}"
    def test_chromadb():
        try:
            from chroma_client import chroma_client
            chroma_client.client.list_collections()
            return True, "Chroma доступна (python client)"
        except Exception as client_exc:
            logger.warning(f"Chroma python client failed, fallback http: {client_exc}")
            response = requests.get(f"{CHROMADB_BASE_URL}/api/v1/heartbeat", timeout=10)
        if response.status_code == 200:
            return True, "Chroma доступна (HTTP)"
        return False, f"Chroma HTTP код {response.status_code}"
    def test_analytics():
        response = requests.get(f"{ANALYTICS_BASE_URL}/api/stats", timeout=10)
        if response.status_code == 200:
            return True, "Аналитика отвечает"
        return False, f"Аналитика код {response.status_code}"
    results = {
        'ollama': measure('ollama', test_ollama),
        'chromadb': measure('chromadb', test_chromadb),
        'analytics': measure('analytics', test_analytics),
    }
    overall_ok = all(service['status'] == 'ok' for service in results.values())
    summary = {
        'status': 'success' if overall_ok else 'degraded',
        'message': 'Все сервисы в норме' if overall_ok else 'Некоторые сервисы недоступны',
        'services': results
    }
    http_status = 200 if overall_ok else 503
    return jsonify(summary), http_status
@app.route('/api/performance')
def performance_stats():

    try:
        return jsonify({
            'search_documents': perf_monitor.get_stats('search_documents'),
            'generate_response': perf_monitor.get_stats('generate_response'),
            'total_requests': sum(len(perf_monitor.metrics[op]) for op in perf_monitor.metrics)
        })
    except Exception as e:
        logger.error(f"Ошибка получения статистики: {e}")
        return jsonify({'error': 'Ошибка получения статистики'}), 500
def sanitize_filename_base(name: str) -> str:
    import re
    import unicodedata
    if not isinstance(name, str):
        return 'file'
    name = unicodedata.normalize('NFC', name)
    name = re.sub(r"[^A-Za-z0-9_\.\-\u0400-\u04FF ]+", "", name)
    name = re.sub(r"\s+", "_", name).strip("._ ")
    return name or 'file'
@app.route('/api/admin/upload-document', methods=['POST'])
def upload_document():

    try:
        if 'file' not in request.files:
            return jsonify({'error': 'Файл не найден'}), 400
        file = request.files['file']
        if file.filename == '':
            return jsonify({'error': 'Файл не выбран'}), 400
        category = request.form.get('category', 'general')
        auto_process = request.form.get('auto_process', 'false').lower() == 'true'
        original_filename = file.filename or 'file'
        logger.info(f"Загрузка файла: оригинальное имя='{original_filename}'")
        name, ext = os.path.splitext(original_filename)
        logger.info(f"После splitext: имя='{name}', расширение='{ext}'")
        safe_name = sanitize_filename_base(name)
        logger.info(f"После sanitize: имя='{safe_name}'")
        ext = ext.lower() if ext else ''
        logger.info(f"Расширение в нижнем регистре: '{ext}'")
        if not ext:
            logger.info("Расширение отсутствует, пытаемся определить по MIME-типу")
            head = file.stream.read(8)
            file.stream.seek(0)
            if head.startswith(b'%PDF'):
                ext = '.pdf'
                logger.info("Определен тип по magic bytes: PDF")
            elif head.startswith(b'PK\x03\x04'):
                try:
                    file.stream.seek(0)
                    head_full = file.stream.read(4096)
                    file.stream.seek(0)
                    if b'word/' in head_full:
                        ext = '.docx'
                    elif b'xl/' in head_full:
                        ext = '.xlsx'
                    elif b'ppt/' in head_full:
                        ext = '.pptx'
                    elif b'OEBPS' in head_full or b'epub' in head_full:
                        ext = '.epub'
                    else:
                        ext = '.zip'
                    logger.info(f"Определен тип по ZIP содержимому: {ext}")
                except:
                    ext = '.zip'
                    file.stream.seek(0)
            elif head.startswith(b'\xd0\xcf\x11\xe0'):
                ext = '.doc'
                logger.info("Определен тип по magic bytes: DOC")
            elif head.startswith(b'{') or head.startswith(b'['):
                ext = '.json'
                logger.info("Определен тип по magic bytes: JSON")
            else:
                try:
                    file.stream.seek(0)
                    sample = file.stream.read(512).decode('utf-8', errors='ignore')
                    file.stream.seek(0)
                    if '<html' in sample.lower() or '<!doctype' in sample.lower():
                        ext = '.html'
                    elif '<?xml' in sample:
                        ext = '.xml'
                    else:
                        ext = '.txt'
                    logger.info(f"Определен текстовый тип: {ext}")
                except:
                    ext = '.bin'
                    file.stream.seek(0)
                    logger.info("Не удалось определить тип, используем .bin")
        os.makedirs('uploads', exist_ok=True)
        if not safe_name:
            safe_name = 'file'
            logger.warning("safe_name пустое, используем 'file'")
        candidate = f"{safe_name}{ext}"
        file_path = os.path.join('uploads', candidate)
        if os.path.exists(file_path):
            import uuid
            candidate = f"{safe_name}_{uuid.uuid4().hex[:8]}{ext}"
            file_path = os.path.join('uploads', candidate)
            logger.info(f"Файл существует, добавлен UUID: {candidate}")
        filename = candidate
        logger.info(f"Финальное имя файла: '{filename}'")
        file.save(file_path)
        if auto_process and os.path.isfile(file_path):
            try:
                from process_documents import process_single_document
                result = process_single_document(file_path, category)
                if result:
                    logger.info(f"Документ {filename} успешно обработан и добавлен в ChromaDB")
                else:
                    logger.error(f"Документ {filename} не удалось обработать (проверьте логи)")
                    return jsonify({
                        'error': 'Документ загружен, но не удалось обработать и добавить в ChromaDB. Проверьте доступность сервисов (Ollama, ChromaDB).',
                        'filename': filename,
                        'category': category,
                        'processed': False
                    }), 500
            except Exception as e:
                logger.error(f"Ошибка обработки документа {filename}: {e}")
                import traceback
                logger.error(traceback.format_exc())
                return jsonify({'error': f'Ошибка обработки документа: {e}'}), 500
        return jsonify({
            'message': 'Документ успешно загружен',
            'filename': filename,
            'category': category,
            'processed': auto_process
        })
    except Exception as e:
        logger.error(f"Ошибка загрузки документа: {e}")
        import traceback
        logger.error(traceback.format_exc())
        return jsonify({'error': f'Ошибка загрузки: {str(e)}'}), 500
@app.route('/api/admin/documents')
def get_documents():

    try:
        documents = []
        uploads_dir = 'uploads'
        if os.path.exists(uploads_dir):
            for filename in os.listdir(uploads_dir):
                file_path = os.path.join(uploads_dir, filename)
                if os.path.isfile(file_path):
                    stat = os.stat(file_path)
                    documents.append({
                        'id': filename,
                        'name': filename,
                        'size': stat.st_size,
                        'created': stat.st_ctime,
                        'modified': stat.st_mtime
                    })
        return jsonify({'documents': documents})
    except Exception as e:
        logger.error(f"Ошибка получения документов: {e}")
        return jsonify({'error': 'Ошибка получения документов'}), 500
@app.route('/api/admin/documents/<document_id>', methods=['DELETE'])
def delete_document(document_id):

    try:
        file_path = os.path.join('uploads', document_id)
        if os.path.exists(file_path):
            os.remove(file_path)
            logger.info(f"Документ {document_id} удален")
            return jsonify({'message': 'Документ удален'})
        else:
            return jsonify({'error': 'Документ не найден'}), 404
    except Exception as e:
        logger.error(f"Ошибка удаления документа {document_id}: {e}")
        return jsonify({'error': 'Ошибка удаления документа'}), 500
@app.route('/api/admin/clear-chromadb', methods=['POST'])
def clear_chromadb():

    try:
        from chroma_client import chroma_client
        collection = chroma_client.get_collection()
        count_before = collection.count()
        logger.info(f"Начинаем очистку ChromaDB. Документов: {count_before}")
        if count_before > 0:
            all_data = collection.get()
            all_ids = all_data['ids']
            batch_size = 100
            for i in range(0, len(all_ids), batch_size):
                batch = all_ids[i:i+batch_size]
                collection.delete(ids=batch)
                logger.info(f"Удалено {min(i+batch_size, len(all_ids))}/{len(all_ids)}")
            count_after = collection.count()
            logger.info(f"Очистка завершена. Документов осталось: {count_after}")
            return jsonify({
                'message': 'ChromaDB очищена',
                'deleted': len(all_ids),
                'remaining': count_after
            })
        else:
            return jsonify({'message': 'ChromaDB уже пуста', 'deleted': 0, 'remaining': 0})
    except Exception as e:
        logger.error(f"Ошибка очистки ChromaDB: {e}")
        return jsonify({'error': 'Ошибка очистки ChromaDB'}), 500
@app.route('/api/debug/search-test')
def debug_search_test():

    try:
        from proper_retriever import ProperRetriever
        query = request.args.get('q', '')
        k = int(request.args.get('k', 10))
        if not query:
            return jsonify({'error': 'Не указан запрос (параметр q)'}), 400
        from proper_retriever import proper_retriever
        retriever = proper_retriever
        documents = retriever.search(query)
        target_id = request.args.get('target', '')
        found_position = None
        top_results = []
        all_results = []
        if documents:
            for i, doc in enumerate(documents):
                metadata = doc.get('metadata', {})
                source = metadata.get('source', f'unknown_{i}')
                chunk_index = metadata.get('chunk_index', i)
                doc_id = f"{source}_chunk_{chunk_index}"
                text = doc.get('content', '')
                dist = doc.get('distance', 0.0)
                result_item = {
                    'position': i + 1,
                    'id': doc_id,
                    'distance': dist,
                    'text_preview': text,
                }
                all_results.append(result_item)
                max_display = retriever.settings.get('max_results', 10)
                if i < max_display:
                    top_results.append(result_item)
                if doc_id == target_id:
                    found_position = i + 1
        target_result = None
        for r in all_results:
            if r['id'] == target_id:
                target_result = r
                break
        return jsonify({
            'query': query,
            'k': k,
            'total_found': len(documents),
            'target_chunk': target_id,
            'target_found': found_position is not None,
            'target_position': found_position,
            'target_result': target_result,
            'top_results': top_results
        })
    except Exception as e:
        logger.error(f"Ошибка debug search: {e}")
        import traceback
        logger.error(traceback.format_exc())
        return jsonify({'error': str(e)}), 500
@app.route('/api/admin/documents/<document_id>/process', methods=['POST'])
def process_document(document_id):

    try:
        import urllib.parse
        document_id = urllib.parse.unquote(document_id)
        file_path = os.path.join('uploads', document_id)
        if not os.path.exists(file_path):
            return jsonify({'error': 'Документ не найден'}), 404
        if not os.path.isfile(file_path):
            return jsonify({'error': 'Указан не файл, а директория'}), 400
        if request.is_json and request.json:
            category = request.json.get('category', 'general')
        elif request.form:
            category = request.form.get('category', 'general')
        else:
            category = 'general'
        from process_documents import process_single_document
        result = process_single_document(file_path, category)
        if result:
            logger.info(f"Документ {document_id} успешно обработан и добавлен в ChromaDB")
            return jsonify({'message': 'Документ успешно обработан и добавлен в ChromaDB'})
        else:
            logger.error(f"Документ {document_id} не удалось обработать")
            return jsonify({'error': 'Документ не удалось обработать. Проверьте логи для подробностей.'}), 500
    except Exception as e:
        logger.error(f"Ошибка обработки документа {document_id}: {e}")
        return jsonify({'error': f'Ошибка обработки: {e}'}), 500
@app.route('/api/admin/parser-settings')
def get_parser_settings():

    try:
        settings_file = 'parser_settings.json'
        if os.path.exists(settings_file):
            with open(settings_file, 'r', encoding='utf-8') as f:
                settings = json.load(f)
        else:
            settings = {
                'chunk_size': 1000,
                'chunk_overlap': 200,
                'enable_metadata': True,
                'language': 'ru'
            }
        return jsonify(settings)
    except Exception as e:
        logger.error(f"Ошибка получения настроек парсера: {e}")
        return jsonify({'error': 'Ошибка получения настроек'}), 500
@app.route('/api/admin/parser-settings', methods=['POST'])
def save_parser_settings():

    try:
        settings = request.json
        settings_file = 'parser_settings.json'
        with open(settings_file, 'w', encoding='utf-8') as f:
            json.dump(settings, f, ensure_ascii=False, indent=2)
        logger.info("Настройки парсера сохранены")
        return jsonify({'message': 'Настройки сохранены'})
    except Exception as e:
        logger.error(f"Ошибка сохранения настроек парсера: {e}")
        return jsonify({'error': 'Ошибка сохранения настроек'}), 500
def _crawl_website_task(task_id: str, website_url: str, max_pages: int, category: str, delay: float, use_llm: bool):

    try:
        with crawler_tasks_lock:
            crawler_tasks[task_id] = {
                'status': 'running',
                'progress': 0,
                'message': 'Начало скачивания сайта...',
                'started_at': datetime.now().isoformat(),
                'result': None
            }
        logger.info(f"[Task {task_id}] Начало скачивания сайта: {website_url}, максимум страниц: {max_pages}")
        from website_crawler import WebsiteCrawler
        from process_documents import DocumentProcessor
        from pathlib import Path
        import tempfile
        import shutil
        crawler = WebsiteCrawler(website_url, max_pages=max_pages, delay=delay, use_llm=use_llm)
        with crawler_tasks_lock:
            crawler_tasks[task_id]['message'] = 'Скачивание страниц...'
            crawler_tasks[task_id]['progress'] = 10
        pages = crawler.crawl()
        if not pages:
            with crawler_tasks_lock:
                crawler_tasks[task_id]['status'] = 'failed'
                crawler_tasks[task_id]['message'] = 'Не удалось скачать ни одной страницы'
            return
        logger.info(f"[Task {task_id}] Скачано страниц: {len(pages)}")
        with crawler_tasks_lock:
            crawler_tasks[task_id]['message'] = f'Скачано {len(pages)} страниц. Обработка...'
            crawler_tasks[task_id]['progress'] = 30
        temp_dir = tempfile.mkdtemp(prefix='website_crawl_')
        try:
            saved_files = crawler.save_to_files(temp_dir)
            with crawler_tasks_lock:
                crawler_tasks[task_id]['message'] = f'Обработка {len(saved_files)} файлов...'
                crawler_tasks[task_id]['progress'] = 50
            processor = DocumentProcessor()
            processed_count = 0
            failed_count = 0
            total_files = len(saved_files)
            for idx, file_path in enumerate(saved_files):
                try:
                    documents = processor.process_any_file(Path(file_path))
                    if documents:
                        for doc in documents:
                            doc['metadata']['category'] = category
                            doc['metadata']['source'] = 'website_crawl'
                            doc['metadata']['website_url'] = website_url
                        if processor.add_documents_to_chromadb(documents):
                            processed_count += 1
                        else:
                            failed_count += 1
                    else:
                        failed_count += 1
                    progress = 50 + int((idx + 1) / total_files * 50)
                    with crawler_tasks_lock:
                        crawler_tasks[task_id]['progress'] = progress
                        crawler_tasks[task_id]['message'] = f'Обработано {idx + 1}/{total_files} файлов...'
                except Exception as e:
                    logger.error(f"[Task {task_id}] Ошибка обработки файла {file_path}: {e}")
                    failed_count += 1
            shutil.rmtree(temp_dir, ignore_errors=True)
            result = {
                'message': f'Сайт успешно проиндексирован',
                'total_pages': len(pages),
                'processed': processed_count,
                'failed': failed_count,
                'website_url': website_url
            }
            with crawler_tasks_lock:
                crawler_tasks[task_id]['status'] = 'completed'
                crawler_tasks[task_id]['progress'] = 100
                crawler_tasks[task_id]['message'] = 'Завершено успешно'
                crawler_tasks[task_id]['result'] = result
                crawler_tasks[task_id]['completed_at'] = datetime.now().isoformat()
            logger.info(f"[Task {task_id}] Задача завершена успешно")
        except Exception as e:
            shutil.rmtree(temp_dir, ignore_errors=True)
            raise
    except Exception as e:
        logger.error(f"[Task {task_id}] Ошибка скачивания сайта: {e}")
        import traceback
        logger.error(traceback.format_exc())
        with crawler_tasks_lock:
            crawler_tasks[task_id]['status'] = 'failed'
            crawler_tasks[task_id]['message'] = f'Ошибка: {str(e)}'
            crawler_tasks[task_id]['completed_at'] = datetime.now().isoformat()
@app.route('/api/admin/crawl-website', methods=['POST'])
def crawl_website():

    try:
        data = request.json if request.is_json else request.form
        website_url = data.get('url', '').strip()
        max_pages = int(data.get('max_pages', 50))
        category = data.get('category', 'website')
        delay = float(data.get('delay', 1.0))
        if not website_url:
            return jsonify({'error': 'URL не указан'}), 400
        from urllib.parse import urlparse
        parsed = urlparse(website_url)
        if not parsed.scheme or not parsed.netloc:
            return jsonify({'error': 'Некорректный URL'}), 400
        use_llm = data.get('use_llm', True)
        task_id = str(uuid.uuid4())
        thread = threading.Thread(
            target=_crawl_website_task,
            args=(task_id, website_url, max_pages, category, delay, use_llm),
            daemon=True
        )
        thread.start()
        logger.info(f"Запущена фоновая задача скрапинга: {task_id} для {website_url}")
        return jsonify({
            'message': 'Задача скрапинга запущена',
            'task_id': task_id,
            'status': 'running',
            'check_status_url': f'/api/admin/crawl-task/{task_id}'
        }), 202
    except Exception as e:
        logger.error(f"Ошибка запуска задачи скрапинга: {e}")
        import traceback
        logger.error(traceback.format_exc())
        return jsonify({'error': f'Ошибка запуска задачи: {str(e)}'}), 500
@app.route('/api/admin/crawl-task/<task_id>', methods=['GET'])
def get_crawl_task_status(task_id):

    with crawler_tasks_lock:
        task = crawler_tasks.get(task_id)
    if not task:
        return jsonify({'error': 'Задача не найдена'}), 404
    return jsonify(task)
@app.route('/api/admin/parse-url', methods=['POST'])
def parse_single_url_endpoint():

    try:
        data = request.json if request.is_json else request.form
        url = data.get('url', '').strip()
        category = data.get('category', 'website_link')
        if not url:
            return jsonify({'error': 'URL не указан'}), 400
        from website_crawler import WebsiteCrawler
        from process_documents import DocumentProcessor
        from pathlib import Path
        import tempfile
        import shutil
        temp_dir = tempfile.mkdtemp(prefix='single_url_parse_')
        try:
            use_llm = data.get('use_llm', True)
            crawler = WebsiteCrawler(url, max_pages=1, delay=0, check_domain=False, use_llm=use_llm)
            page_data = crawler.parse_single_url(url)
            if not page_data:
                return jsonify({'error': 'Не удалось получить содержимое страницы'}), 500
            file_path = os.path.join(temp_dir, f"{uuid.uuid4()}.html")
            with open(file_path, 'w', encoding='utf-8') as f:
                f.write(page_data['content'])
            processor = DocumentProcessor()
            documents = processor.process_any_file(Path(file_path))
            if documents:
                for doc in documents:
                    doc['metadata']['category'] = category
                    doc['metadata']['source'] = 'single_url_parse'
                    doc['metadata']['original_url'] = url
                if processor.add_documents_to_chromadb(documents):
                    logger.info(f"Страница {url} успешно проиндексирована.")
                    return jsonify({'message': f'Страница {url} успешно проиндексирована', 'url': url, 'category': category})
                else:
                    return jsonify({'error': f'Не удалось добавить страницу {url} в ChromaDB'}), 500
            else:
                return jsonify({'error': f'Не удалось обработать содержимое страницы {url}'}), 500
        finally:
            shutil.rmtree(temp_dir, ignore_errors=True)
    except Exception as e:
        logger.error(f"Ошибка парсинга URL: {e}")
        import traceback
        logger.error(traceback.format_exc())
        return jsonify({'error': f'Ошибка парсинга URL: {str(e)}'}), 500
@app.route('/api/admin/delete-crawler-data', methods=['POST'])
def delete_crawler_data():

    try:
        from chroma_client import chroma_client
        data = request.json if request.is_json else request.form
        source_type = data.get('source_type', 'all')
        deleted_counts = {}
        total_deleted = 0
        if source_type == 'all':
            crawler_sources = ['website_crawl', 'single_url_parse', 'multiple_urls_parse']
            for source in crawler_sources:
                count = chroma_client.delete_documents_by_metadata({'source': source})
                deleted_counts[source] = count
                total_deleted += count
        else:
            count = chroma_client.delete_documents_by_metadata({'source': source_type})
            deleted_counts[source_type] = count
            total_deleted = count
        logger.info(f"Удалено {total_deleted} документов краулера")
        return jsonify({
            'message': f'Удалено {total_deleted} документов краулера',
            'deleted': deleted_counts,
            'total_deleted': total_deleted
        })
    except Exception as e:
        logger.error(f"Ошибка удаления данных краулера: {e}")
        import traceback
        logger.error(traceback.format_exc())
        return jsonify({'error': f'Ошибка удаления данных: {str(e)}'}), 500
@app.route('/api/admin/crawler-data-stats', methods=['GET'])
def get_crawler_data_stats():

    try:
        from chroma_client import chroma_client
        counts = chroma_client.get_crawler_documents_count()
        return jsonify({
            'stats': counts,
            'total_crawler_documents': counts.get('total', 0)
        })
    except Exception as e:
        logger.error(f"Ошибка получения статистики: {e}")
        return jsonify({'error': f'Ошибка получения статистики: {str(e)}'}), 500
@app.route('/api/admin/parse-urls', methods=['POST'])
def parse_multiple_urls_endpoint():

    try:
        data = request.json if request.is_json else request.form
        urls_input = data.get('urls', '')
        category = data.get('category', 'website_links')
        delay = float(data.get('delay', 1.0))
        if isinstance(urls_input, str):
            urls = [u.strip() for u in re.split(r'[\n,]', urls_input) if u.strip()]
        elif isinstance(urls_input, list):
            urls = [u.strip() for u in urls_input if u.strip()]
        else:
            return jsonify({'error': 'Некорректный формат списка URL'}), 400
        if not urls:
            return jsonify({'error': 'Список URL пуст'}), 400
        from website_crawler import WebsiteCrawler
        from process_documents import DocumentProcessor
        from pathlib import Path
        import tempfile
        import shutil
        temp_dir = tempfile.mkdtemp(prefix='multiple_urls_parse_')
        processed_count = 0
        failed_count = 0
        try:
            use_llm = data.get('use_llm', True)
            crawler = WebsiteCrawler(urls[0], max_pages=len(urls), delay=delay, check_domain=False, use_llm=use_llm)
            for url in urls:
                page_data = crawler.parse_single_url(url)
                if page_data:
                    file_path = os.path.join(temp_dir, f"{uuid.uuid4()}.html")
                    with open(file_path, 'w', encoding='utf-8') as f:
                        f.write(page_data['content'])
                    processor = DocumentProcessor()
                    documents = processor.process_any_file(Path(file_path))
                    if documents:
                        for doc in documents:
                            doc['metadata']['category'] = category
                            doc['metadata']['source'] = 'multiple_urls_parse'
                            doc['metadata']['original_url'] = url
                        if processor.add_documents_to_chromadb(documents):
                            processed_count += 1
                        else:
                            failed_count += 1
                    else:
                        failed_count += 1
                else:
                    failed_count += 1
                time.sleep(delay)
            return jsonify({
                'message': f'Обработка списка URL завершена',
                'total_urls': len(urls),
                'processed': processed_count,
                'failed': failed_count,
                'category': category
            })
        finally:
            shutil.rmtree(temp_dir, ignore_errors=True)
    except Exception as e:
        logger.error(f"Ошибка парсинга списка URL: {e}")
        import traceback
        logger.error(traceback.format_exc())
        return jsonify({'error': f'Ошибка парсинга списка URL: {str(e)}'}), 500
@app.route('/api/admin/retriever-settings')
def get_retriever_settings():

    try:
        settings_file = 'retriever_settings.json'
        if os.path.exists(settings_file):
            with open(settings_file, 'r', encoding='utf-8') as f:
                settings = json.load(f)
        else:
            settings = {
                'search_k': SEARCH_K,
                'similarity_threshold': 0.2,
                'max_results': 15,
                'search_type': 'similarity',
                'mmr_lambda': 0.7,
                'rerank_results': 'simple',
                'filter_by_category': False,
                'filter_by_date': False,
                'filter_by_size': False
            }
        return jsonify(settings)
    except Exception as e:
        logger.error(f"Ошибка получения настроек ретривера: {e}")
        return jsonify({'error': 'Ошибка получения настроек'}), 500
@app.route('/api/admin/retriever-settings', methods=['POST'])
def save_retriever_settings():

    try:
        settings = request.json
        settings_file = 'retriever_settings.json'
        with open(settings_file, 'w', encoding='utf-8') as f:
            json.dump(settings, f, ensure_ascii=False, indent=2)
        logger.info("Настройки ретривера сохранены")
        return jsonify({'message': 'Настройки сохранены'})
    except Exception as e:
        logger.error(f"Ошибка сохранения настроек ретривера: {e}")
        return jsonify({'error': 'Ошибка сохранения настроек'}), 500
@app.route('/api/admin/backup', methods=['POST'])
def create_backup():

    try:
        import shutil
        from datetime import datetime
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        backup_name = f"backup_{timestamp}"
        backup_dir = f"backups/{backup_name}"
        os.makedirs(backup_dir, exist_ok=True)
        items_to_backup = [
            'uploads',
            'data',
            'parser_settings.json',
            'retriever_settings.json',
            'documents.db'
        ]
        for item in items_to_backup:
            if os.path.exists(item):
                if os.path.isdir(item):
                    shutil.copytree(item, os.path.join(backup_dir, item))
                else:
                    shutil.copy2(item, backup_dir)
        backup_info = {
            'name': backup_name,
            'created': datetime.now().isoformat(),
            'size': sum(os.path.getsize(os.path.join(backup_dir, f))
                       for f in os.listdir(backup_dir)
                       if os.path.isfile(os.path.join(backup_dir, f))),
            'items': items_to_backup
        }
        with open(os.path.join(backup_dir, 'backup_info.json'), 'w') as f:
            json.dump(backup_info, f, indent=2)
        logger.info(f"Backup создан: {backup_name}")
        return jsonify({
            'message': 'Backup успешно создан',
            'backup_name': backup_name,
            'size': backup_info['size']
        })
    except Exception as e:
        logger.error(f"Ошибка создания backup: {e}")
        return jsonify({'error': f'Ошибка создания backup: {e}'}), 500
@app.route('/api/admin/backups')
def list_backups():

    try:
        backups = []
        backups_dir = 'backups'
        if os.path.exists(backups_dir):
            for backup_name in os.listdir(backups_dir):
                backup_path = os.path.join(backups_dir, backup_name)
                if os.path.isdir(backup_path):
                    info_file = os.path.join(backup_path, 'backup_info.json')
                    if os.path.exists(info_file):
                        with open(info_file, 'r') as f:
                            backup_info = json.load(f)
                        backups.append(backup_info)
                    else:
                        stat = os.stat(backup_path)
                        backups.append({
                            'name': backup_name,
                            'created': datetime.fromtimestamp(stat.st_ctime).isoformat(),
                            'size': sum(os.path.getsize(os.path.join(backup_path, f))
                                       for f in os.listdir(backup_path)
                                       if os.path.isfile(os.path.join(backup_path, f))),
                            'items': []
                        })
        return jsonify({'backups': sorted(backups, key=lambda x: x['created'], reverse=True)})
    except Exception as e:
        logger.error(f"Ошибка получения списка backup'ов: {e}")
        return jsonify({'error': 'Ошибка получения списка backup\'ов'}), 500
@app.route('/api/admin/backup/<backup_name>/restore', methods=['POST'])
def restore_backup(backup_name):

    try:
        import shutil
        backup_dir = f"backups/{backup_name}"
        if not os.path.exists(backup_dir):
            return jsonify({'error': 'Backup не найден'}), 404
        info_file = os.path.join(backup_dir, 'backup_info.json')
        if os.path.exists(info_file):
            with open(info_file, 'r') as f:
                backup_info = json.load(f)
        else:
            backup_info = {'items': []}
        restored_items = []
        for item in backup_info.get('items', []):
            source = os.path.join(backup_dir, item)
            if os.path.exists(source):
                if os.path.isdir(source):
                    if os.path.exists(item):
                        shutil.rmtree(item)
                    shutil.copytree(source, item)
                else:
                    shutil.copy2(source, item)
                restored_items.append(item)
        logger.info(f"Backup восстановлен: {backup_name}")
        return jsonify({
            'message': 'Backup успешно восстановлен',
            'restored_items': restored_items
        })
    except Exception as e:
        logger.error(f"Ошибка восстановления backup {backup_name}: {e}")
        return jsonify({'error': f'Ошибка восстановления: {e}'}), 500
@app.route('/api/admin/backup/<backup_name>', methods=['DELETE'])
def delete_backup(backup_name):

    try:
        import shutil
        backup_dir = f"backups/{backup_name}"
        if not os.path.exists(backup_dir):
            return jsonify({'error': 'Backup не найден'}), 404
        shutil.rmtree(backup_dir)
        logger.info(f"Backup удален: {backup_name}")
        return jsonify({'message': 'Backup удален'})
    except Exception as e:
        logger.error(f"Ошибка удаления backup {backup_name}: {e}")
        return jsonify({'error': 'Ошибка удаления backup'}), 500
@app.route('/api/admin/optimize', methods=['POST'])
def optimize_system():

    try:
        optimization_results = []
        temp_dirs = ['temp', 'cache', '__pycache__']
        cleaned_files = 0
        for temp_dir in temp_dirs:
            if os.path.exists(temp_dir):
                for root, dirs, files in os.walk(temp_dir):
                    for file in files:
                        if file.endswith(('.pyc', '.pyo', '.tmp')):
                            os.remove(os.path.join(root, file))
                            cleaned_files += 1
        optimization_results.append(f"Очищено временных файлов: {cleaned_files}")
        if os.path.exists('documents.db'):
            import sqlite3
            conn = sqlite3.connect('documents.db')
            cursor = conn.cursor()
            cursor.execute("PRAGMA analysis_limit=1000")
            cursor.execute("ANALYZE")
            cursor.execute("SELECT COUNT(*) FROM documents")
            doc_count = cursor.fetchone()[0]
            cursor.execute("SELECT name FROM sqlite_master WHERE type='table' AND name='chunks'")
            chunks_table_exists = cursor.fetchone() is not None
            if chunks_table_exists:
                cursor.execute("SELECT COUNT(*) FROM chunks")
                chunk_count = cursor.fetchone()[0]
            else:
                chunk_count = 0
            conn.close()
            optimization_results.append(f"База данных оптимизирована: {doc_count} документов, {chunk_count} чанков")
        log_files = []
        for root, dirs, files in os.walk('.'):
            for file in files:
                if file.endswith('.log') and os.path.getsize(os.path.join(root, file)) > 10 * 1024 * 1024:
                    log_files.append(os.path.join(root, file))
        optimization_results.append(f"Найдено больших лог-файлов: {len(log_files)}")
        logger.info("Система оптимизирована")
        return jsonify({
            'message': 'Оптимизация завершена',
            'results': optimization_results
        })
    except Exception as e:
        logger.error(f"Ошибка оптимизации системы: {e}")
        return jsonify({'error': f'Ошибка оптимизации: {e}'}), 500
@app.route('/api/admin/restart-service', methods=['POST'])
def restart_service():

    try:
        service = request.json.get('service', 'all')
        container_mapping = {
            'rag-chat-api': os.getenv("RAG_CHAT_CONTAINER", "autogen-dev-rag-chat-api"),
            'ollama': os.getenv("OLLAMA_CONTAINER", None),
            'chromadb': os.getenv("CHROMADB_CONTAINER", "shared-chromadb"),
            'analytics': os.getenv("ANALYTICS_CONTAINER", "autogen-dev-analytics"),
            'reranker': "autogen-dev-reranker"
        }
        if service == 'all':
            import subprocess
            compose_path = os.getenv('COMPOSE_PATH', '/opt/autogen/development')
            result = subprocess.run(['docker-compose', '-f', 'docker-compose.dev.yml', 'restart'],
                                  capture_output=True, text=True, cwd=compose_path)
            if result.returncode == 0:
                logger.info("Все сервисы перезапущены")
                return jsonify({'message': 'Все сервисы перезапущены'})
            else:
                return jsonify({'error': f'Ошибка перезапуска: {result.stderr}'}), 500
        else:
            import docker
            try:
                client = docker.from_env()
                container_name = container_mapping.get(service, service)
                if container_name is None:
                    return jsonify({'error': f'Сервис {service} работает на внешнем сервере, перезапуск через Docker недоступен'}), 400
                container = client.containers.get(container_name)
                container.restart()
                logger.info(f"Сервис {service} перезапущен")
                return jsonify({'message': f'Сервис {service} перезапущен'})
            except docker.errors.NotFound:
                available_containers = [c.name for c in client.containers.list(all=True)]
                return jsonify({'error': f'Контейнер {container_name} не найден. Доступные контейнеры: {", ".join(available_containers)}'}), 404
            except Exception as e:
                logger.error(f"Ошибка перезапуска сервиса {service}: {e}")
                return jsonify({'error': f'Ошибка перезапуска {service}: {e}'}), 500
    except Exception as e:
        logger.error(f"Ошибка перезапуска сервиса: {e}")
        return jsonify({'error': f'Ошибка перезапуска: {e}'}), 500
@app.route('/api/admin/logs/<service>')
def get_service_logs(service):

    try:
        import docker
        container_mapping = {
            'rag-chat-api': os.getenv("RAG_CHAT_CONTAINER", "autogen-dev-rag-chat-api"),
            'ollama': os.getenv("OLLAMA_CONTAINER", None),
            'chromadb': os.getenv("CHROMADB_CONTAINER", "shared-chromadb"),
            'analytics': os.getenv("ANALYTICS_CONTAINER", "autogen-dev-analytics"),
            'reranker': "autogen-dev-reranker"
        }
        container_name = container_mapping.get(service, service)
        if container_name is None:
            return f"Сервис {service} работает на внешнем сервере, логи недоступны через Docker", 200, {'Content-Type': 'text/plain; charset=utf-8'}
        try:
            client = docker.from_env()
            container = client.containers.get(container_name)
            logs = container.logs(tail=200, timestamps=True).decode('utf-8')
            if logs:
                return logs, 200, {'Content-Type': 'text/plain; charset=utf-8'}
            else:
                return f"Логи для сервиса {service} не найдены", 200, {'Content-Type': 'text/plain; charset=utf-8'}
        except docker.errors.NotFound:
            return f"Контейнер {container_name} не найден. Доступные контейнеры: {', '.join([c.name for c in client.containers.list(all=True)])}", 404, {'Content-Type': 'text/plain; charset=utf-8'}
        except Exception as e:
            logger.error(f"Ошибка получения логов сервиса {service}: {e}")
            return f"Ошибка получения логов сервиса {service}: {e}", 500, {'Content-Type': 'text/plain; charset=utf-8'}
    except Exception as e:
        logger.error(f"Ошибка получения логов сервиса {service}: {e}")
        return f"Ошибка получения логов сервиса {service}: {e}", 500, {'Content-Type': 'text/plain; charset=utf-8'}
@app.route('/api/admin/clear-logs', methods=['POST'])
def clear_logs():

    try:
        import subprocess
        services = [
            os.getenv("RAG_CHAT_CONTAINER", "autogen-dev-rag-chat-api"),
            os.getenv('ANALYTICS_HOST', 'rag-analytics'),
            os.getenv("CHROMADB_CONTAINER", "shared-chromadb")
        ]
        cleared_services = []
        for service in services:
            try:
                result = subprocess.run(['docker', 'logs', '--tail', '0', service],
                                      capture_output=True, text=True, timeout=10)
                if result.returncode == 0:
                    cleared_services.append(service)
            except:
                pass
        log_files_cleared = 0
        for root, dirs, files in os.walk('.'):
            for file in files:
                if file.endswith('.log'):
                    log_path = os.path.join(root, file)
                    try:
                        with open(log_path, 'w') as f:
                            f.write('')
                        log_files_cleared += 1
                    except:
                        pass
        logger.info("Логи очищены")
        return jsonify({
            'message': 'Логи очищены',
            'cleared_services': cleared_services,
            'cleared_files': log_files_cleared
        })
    except Exception as e:
        logger.error(f"Ошибка очистки логов: {e}")
        return jsonify({'error': f'Ошибка очистки логов: {e}'}), 500
@app.route('/api/admin/reset', methods=['POST'])
def reset_system():

    try:
        import docker
        import shutil
        try:
            client = docker.from_env()
            containers = client.containers.list(filters={'name': 'dev-'})
            for container in containers:
                container.stop()
                logger.info(f"Остановлен контейнер: {container.name}")
        except Exception as e:
            logger.warning(f"Не удалось остановить контейнеры через Docker API: {e}")
        try:
            shutil.rmtree('/app/data/chroma_db', ignore_errors=True)
            os.makedirs('/app/data/chroma_db', exist_ok=True)
        except:
            pass
        try:
            shutil.rmtree('/app/data/uploads', ignore_errors=True)
            os.makedirs('/app/data/uploads', exist_ok=True)
        except:
            pass
        try:
            shutil.rmtree('/app/data/backups', ignore_errors=True)
            os.makedirs('/app/data/backups', exist_ok=True)
        except:
            pass
        try:
            client = docker.from_env()
            for container in containers:
                container.start()
                logger.info(f"Запущен контейнер: {container.name}")
        except Exception as e:
            logger.warning(f"Не удалось перезапустить контейнеры через Docker API: {e}")
        logger.info("Система сброшена к начальным настройкам")
        return jsonify({
            'message': 'Система успешно сброшена к начальным настройкам',
            'actions': [
                'Остановлены все контейнеры',
                'Очищена база данных ChromaDB',
                'Удалены загруженные документы',
                'Очищены backup\'ы',
                'Перезапущены контейнеры'
            ]
        })
    except Exception as e:
        logger.error(f"Ошибка сброса системы: {e}")
        return jsonify({'error': f'Ошибка сброса системы: {e}'}), 500
@app.route('/api/todo')
def api_todo():

    try:
        with open('TODO.md', 'r', encoding='utf-8') as f:
            content = f.read()
        html_content = f
        return html_content, 200, {'Content-Type': 'text/html; charset=utf-8'}
    except FileNotFoundError:
        return '<p>TODO.md файл не найден</p>', 404
    except Exception as e:
        logger.error(f"Ошибка загрузки TODO: {e}")
        return f'<p>Ошибка загрузки TODO: {e}</p>', 500
@app.route('/api/problems')
def api_problems():

    try:
        with open('PROBLEMS_REGISTRY.md', 'r', encoding='utf-8') as f:
            content = f.read()
        html_content = f
        return html_content, 200, {'Content-Type': 'text/html; charset=utf-8'}
    except FileNotFoundError:
        return '<p>PROBLEMS_REGISTRY.md файл не найден</p>', 404
    except Exception as e:
        logger.error(f"Ошибка загрузки проблем: {e}")
        return f'<p>Ошибка загрузки проблем: {e}</p>', 500
@app.route('/api/test-analysis')
def api_test_analysis():

    try:
        paths_to_try = [
            'TEST_ANALYSIS.md',
            '../tmp/ANALYSIS.md',
            '/app/data/ANALYSIS.md',
        ]
        content = None
        for path in paths_to_try:
            try:
                with open(path, 'r', encoding='utf-8') as f:
                    content = f.read()
                    break
            except FileNotFoundError:
                continue
        if content is None:
            return '''<div style="font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; padding: 40px; text-align: center;">
                <h2 style="color: #9b59b6;">📊 Анализ тестов</h2>
                <p>Файл ANALYSIS.md не найден.</p>
                <p style="color: #666;">Разместите файл TEST_ANALYSIS.md в директории приложения или /app/data/ANALYSIS.md</p>
            </div>''', 404
        html_content = f"""
        <div style="font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; line-height: 1.6; color: #333;">
            <h1 style="color: #9b59b6; border-bottom: 2px solid #9b59b6; padding-bottom: 10px;">📊 АНАЛИЗ ТЕСТОВ БОТА - Development Environment</h1>
            <div style="white-space: pre-wrap; background: #f8f9fa; padding: 20px; border-radius: 8px; border-left: 4px solid #9b59b6;">
{content}
            </div>
        </div>
        """
        return html_content, 200, {'Content-Type': 'text/html; charset=utf-8'}
    except Exception as e:
        logger.error(f"Ошибка загрузки анализа тестов: {e}")
        return f'<p>Ошибка загрузки анализа тестов: {e}</p>', 500
STICKY_NOTES_FILE = '/app/data/sticky_notes.json'
def load_sticky_notes_data():

    try:
        if os.path.exists(STICKY_NOTES_FILE):
            with open(STICKY_NOTES_FILE, 'r', encoding='utf-8') as f:
                return json.load(f)
        return {}
    except Exception as e:
        logger.error(f"Ошибка загрузки стикеров: {e}")
        return {}
def save_sticky_notes_data(data):

    try:
        os.makedirs(os.path.dirname(STICKY_NOTES_FILE), exist_ok=True)
        with open(STICKY_NOTES_FILE, 'w', encoding='utf-8') as f:
            json.dump(data, f, ensure_ascii=False, indent=2)
        return True
    except Exception as e:
        logger.error(f"Ошибка сохранения стикеров: {e}")
        return False
@app.route('/api/sticky-notes/<section>', methods=['GET'])
def get_sticky_notes(section):

    try:
        all_notes = load_sticky_notes_data()
        section_notes = all_notes.get(section, {})
        return jsonify(section_notes), 200
    except Exception as e:
        logger.error(f"Ошибка получения стикеров: {e}")
        return jsonify({'error': str(e)}), 500
@app.route('/api/sticky-notes/<section>', methods=['POST'])
def save_sticky_notes(section):

    try:
        notes = request.json
        all_notes = load_sticky_notes_data()
        all_notes[section] = notes
        if save_sticky_notes_data(all_notes):
            return jsonify({'status': 'ok'}), 200
        else:
            return jsonify({'error': 'Failed to save'}), 500
    except Exception as e:
        logger.error(f"Ошибка сохранения стикеров: {e}")
        return jsonify({'error': str(e)}), 500
@app.route('/api/sticky-notes/<section>/<note_id>', methods=['DELETE'])
def delete_sticky_note(section, note_id):

    try:
        all_notes = load_sticky_notes_data()
        if section in all_notes and note_id in all_notes[section]:
            del all_notes[section][note_id]
            save_sticky_notes_data(all_notes)
            return jsonify({'status': 'ok'}), 200
        return jsonify({'error': 'Note not found'}), 404
    except Exception as e:
        logger.error(f"Ошибка удаления стикера: {e}")
        return jsonify({'error': str(e)}), 500
@app.errorhandler(RequestEntityTooLarge)
def handle_file_too_large(e):

    logger.error(f"Файл слишком большой: {e}")
    return jsonify({
        'error': f'Файл слишком большой. Максимальный размер: {app.config.get("MAX_CONTENT_LENGTH", 0) / (1024 * 1024):.0f} MB'
    }), 413
@app.route('/static/<path:filename>')
def static_files(filename):

    return send_from_directory('static', filename)
if __name__ == '__main__':
    logger.info("Запуск RAG Chat API...")
    logger.info(f"Ollama URL: {OLLAMA_BASE_URL_FIRST} (из {len(OLLAMA_BASE_URL.split(',')) if OLLAMA_BASE_URL else 0} серверов)")
    logger.info(f"ChromaDB URL: {CHROMADB_BASE_URL}")
    logger.info(f"Analytics URL: {ANALYTICS_BASE_URL}")
    logger.info(f"Model: {MODEL_NAME}")
    logger.info(f"Temperature: {TEMPERATURE}")
    logger.info(f"Search K: {SEARCH_K}")
    app.run(host='0.0.0.0', port=int(os.getenv('FLASK_PORT', '9004')), debug=False)