import os
import logging
import sqlite3
import uuid
import json
from typing import Dict, Any, List, Optional
from pathlib import Path
from document_parsers import parser_manager
from embedding_service import embedding_service
logger = logging.getLogger(__name__)
class FileManager:

    def __init__(self, upload_folder: str = 'uploads', db_path: str = 'documents.db'):
        self.upload_folder = upload_folder
        self.db_path = db_path
        self.allowed_extensions = parser_manager.get_supported_extensions()
        os.makedirs(upload_folder, exist_ok=True)
        self._init_database()
    def _init_database(self):

        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute()
        conn.commit()
        conn.close()
    def is_allowed_file(self, filename: str) -> bool:

        return '.' in filename and \
               os.path.splitext(filename)[1].lower() in self.allowed_extensions
    def save_file(self, file, category: str = 'general') -> Dict[str, Any]:

        if not file or not file.filename:
            raise ValueError("Файл не выбран")
        if not self.is_allowed_file(file.filename):
            raise ValueError(f"Неподдерживаемый формат файла: {file.filename}")
        file_id = str(uuid.uuid4())
        original_filename = file.filename
        file_extension = os.path.splitext(original_filename)[1].lower()
        saved_filename = f"{file_id}{file_extension}"
        file_path = os.path.join(self.upload_folder, saved_filename)
        file.save(file_path)
        file_size = os.path.getsize(file_path)
        file_type = file_extension.lower()
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute('''
            INSERT INTO documents 
            (id, filename, original_filename, file_path, file_size, file_type, category)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ''', (file_id, saved_filename, original_filename, file_path, file_size, file_type, category))
        conn.commit()
        conn.close()
        return {
            'id': file_id,
            'filename': original_filename,
            'file_path': file_path,
            'size': file_size,
            'type': file_type,
            'category': category
        }
    def process_document(self, document_id: str) -> Dict[str, Any]:

        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute('SELECT * FROM documents WHERE id = ?', (document_id,))
        document = cursor.fetchone()
        if not document:
            conn.close()
            raise ValueError("Документ не найден")
        try:
            parsed_data = parser_manager.parse_document(document[3])
            cursor.execute('''
                UPDATE documents 
                SET content = ?, metadata = ?, processed = TRUE, status = 'processed'
                WHERE id = ?
            ''', (
                parsed_data['content'],
                str(parsed_data['metadata']),
                document_id
            ))
            conn.commit()
            conn.close()
            try:
                self._save_to_chromadb(document_id, parsed_data['content'], parsed_data['metadata'], document[6])
                logger.info(f"Документ {document_id} успешно сохранен в ChromaDB")
            except Exception as chroma_error:
                logger.error(f"Ошибка сохранения в ChromaDB: {chroma_error}")
            return {
                'id': document_id,
                'content': parsed_data['content'],
                'metadata': parsed_data['metadata'],
                'status': 'processed'
            }
        except Exception as e:
            cursor.execute('''
                UPDATE documents 
                SET status = 'error', processed = FALSE
                WHERE id = ?
            ''', (document_id,))
            conn.commit()
            conn.close()
            logger.error(f"Ошибка обработки документа {document_id}: {e}")
            raise
    def get_document(self, document_id: str) -> Optional[Dict[str, Any]]:

        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute('SELECT * FROM documents WHERE id = ?', (document_id,))
        document = cursor.fetchone()
        conn.close()
        if not document:
            return None
        return {
            'id': document[0],
            'filename': document[1],
            'original_filename': document[2],
            'file_path': document[3],
            'file_size': document[4],
            'file_type': document[5],
            'category': document[6],
            'upload_date': document[7],
            'processed': bool(document[8]),
            'content': document[9],
            'metadata': document[10],
            'status': document[11]
        }
    def get_all_documents(self) -> List[Dict[str, Any]]:

        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute('''
            SELECT id, original_filename, file_size, file_type, category, 
                   upload_date, processed, status
            FROM documents 
            ORDER BY upload_date DESC
        ''')
        documents = []
        for row in cursor.fetchall():
            documents.append({
                'id': row[0],
                'filename': row[1],
                'size': row[2],
                'type': row[3],
                'category': row[4],
                'upload_date': row[5],
                'processed': bool(row[6]),
                'status': row[7]
            })
        conn.close()
        return documents
    def delete_document(self, document_id: str) -> bool:

        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute('SELECT file_path FROM documents WHERE id = ?', (document_id,))
        result = cursor.fetchone()
        if not result:
            conn.close()
            return False
        file_path = result[0]
        try:
            if os.path.exists(file_path):
                os.remove(file_path)
        except Exception as e:
            logger.error(f"Ошибка удаления файла {file_path}: {e}")
        cursor.execute('DELETE FROM documents WHERE id = ?', (document_id,))
        conn.commit()
        conn.close()
        return True
    def get_document_content(self, document_id: str) -> Optional[str]:

        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute('SELECT content FROM documents WHERE id = ?', (document_id,))
        result = cursor.fetchone()
        conn.close()
        return result[0] if result else None
    def search_documents(self, query: str, category: Optional[str] = None) -> List[Dict[str, Any]]:

        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        sql = 'SELECT id, original_filename, file_size, file_type, category, upload_date, processed, status FROM documents WHERE (original_filename LIKE ? OR content LIKE ?)'
        params = [f'%{query}%', f'%{query}%']
        if category:
            sql += ' AND category = ?'
            params.append(category)
        sql += ' ORDER BY upload_date DESC'
        cursor.execute(sql, params)
        documents = []
        for row in cursor.fetchall():
            documents.append({
                'id': row[0],
                'filename': row[1],
                'size': row[2],
                'type': row[3],
                'category': row[4],
                'upload_date': row[5],
                'processed': bool(row[6]),
                'status': row[7]
            })
        conn.close()
        return documents
    def _save_to_chromadb(self, document_id: str, content: str, metadata: dict, category: str):

        try:
            from chroma_client import chroma_client
            try:
                with open('parser_settings.json', 'r', encoding='utf-8') as f:
                    parser_settings = json.load(f)
                    chunk_size = parser_settings.get('chunk_size', 1000)
                    chunk_overlap = parser_settings.get('chunk_overlap', 200)
            except:
                chunk_size = 1000
                chunk_overlap = 200
            chunks = []
            for i in range(0, len(content), chunk_size - chunk_overlap):
                chunk = content[i:i + chunk_size]
                if chunk.strip():
                    chunks.append(chunk)
            documents = []
            metadatas = []
            ids = []
            for i, chunk in enumerate(chunks):
                documents.append(chunk)
                metadatas.append({
                    "document_id": document_id,
                    "chunk_index": i,
                    "category": category,
                    "total_chunks": len(chunks),
                    **metadata
                })
                ids.append(f"{document_id}_chunk_{i}")
            success = chroma_client.add_documents(documents, metadatas, ids)
            if success:
                logger.info(f" Документ {document_id} успешно сохранен в ChromaDB ({len(chunks)} чанков)")
            else:
                logger.error(f" Ошибка сохранения документа {document_id} в ChromaDB")
        except Exception as e:
            logger.error(f" Ошибка сохранения в ChromaDB: {e}")
    def _create_embedding(self, text: str) -> list:

        return embedding_service.create_embedding(text)
file_manager = FileManager()