import logging
from chroma_client import chroma_client
from embedding_service import embedding_service
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)
def migrate_to_cosine():

    old_collection_name = "shared_documents"
    new_collection_name = "shared_documents_cosine"
    try:
        old_collection = chroma_client.client.get_collection(old_collection_name)
        count = old_collection.count()
        logger.info(f"=== Миграция коллекции {old_collection_name} на cosine distance ===")
        logger.info(f"Документов для миграции: {count}")
        collections = chroma_client.client.list_collections()
        collection_names = [c.name for c in collections]
        if new_collection_name in collection_names:
            logger.warning(f"Коллекция {new_collection_name} уже существует, удаляем...")
            chroma_client.client.delete_collection(new_collection_name)
        logger.info(f"Создаем новую коллекцию {new_collection_name} с cosine distance...")
        new_collection = chroma_client.client.create_collection(
            name=new_collection_name,
            metadata={
                "description": "RAG documents collection (cosine distance)",
                "hnsw:space": "cosine"
            }
        )
        logger.info(f"✅ Коллекция {new_collection_name} создана")
        batch_size = 100
        total_migrated = 0
        logger.info(f"Начинаем миграцию документов батчами по {batch_size}...")
        for offset in range(0, count, batch_size):
            results = old_collection.get(
                limit=batch_size,
                offset=offset
            )
            if not results['ids']:
                break
            batch_ids = results['ids']
            batch_documents = results['documents']
            batch_metadatas = results['metadatas']
            batch_embeddings = results.get('embeddings', [])
            if not batch_embeddings:
                logger.info(f"Создаем эмбеддинги для батча {offset//batch_size + 1}...")
                batch_embeddings = embedding_service.create_embeddings_batch(batch_documents)
            new_collection.add(
                ids=batch_ids,
                documents=batch_documents,
                metadatas=batch_metadatas,
                embeddings=batch_embeddings
            )
            total_migrated += len(batch_ids)
            logger.info(f"Мигрировано {total_migrated}/{count} документов...")
        logger.info(f"✅ Миграция завершена: {total_migrated} документов")
        new_count = new_collection.count()
        logger.info(f"Новая коллекция содержит {new_count} документов")
        if new_count == count:
            logger.info("✅ Количество документов совпадает!")
        else:
            logger.warning(f"⚠️ Несоответствие: старая коллекция {count}, новая {new_count}")
        return True
    except Exception as e:
        logger.error(f"Ошибка миграции: {e}")
        import traceback
        traceback.print_exc()
        return False
if __name__ == "__main__":
    success = migrate_to_cosine()
    if success:
        print("\n✅ Миграция успешно завершена!")
        print("Следующие шаги:")
        print("1. Обновить chroma_client.py для использования новой коллекции")
        print("2. Обновить формулу similarity в proper_retriever.py")
        print("3. Обновить threshold в retriever_settings.json")
        print("4. Протестировать поиск")
    else:
        print("\n❌ Миграция не удалась. Проверьте логи.")