import os
import json
import logging
import requests
from typing import List, Dict, Any, Optional
logger = logging.getLogger(__name__)
class RerankerClient:

    def __init__(self):
        reranker_urls = os.getenv('RERANKER_SERVICE_URLS', '')
        self.reranker_urls = [url.strip() for url in reranker_urls.split(',') if url.strip()]
        self.current_url_index = 0
        self.timeout = int(os.getenv('RERANKER_TIMEOUT', '60'))
        logger.info(f"Reranker Client инициализирован с {len(self.reranker_urls)} сервисами: {self.reranker_urls}")
    def _get_next_url(self) -> str:

        url = self.reranker_urls[self.current_url_index]
        self.current_url_index = (self.current_url_index + 1) % len(self.reranker_urls)
        return url
    def rerank(self, documents: List[Dict[str, Any]], query: str) -> Optional[List[Dict[str, Any]]]:

        if not documents or not query:
            return documents
        max_docs = int(os.getenv('RERANKER_MAX_DOCS', '100'))
        if len(documents) > max_docs:
            logger.warning(f"Количество документов ({len(documents)}) превышает максимум ({max_docs}), ограничиваем до {max_docs}")
            documents = documents[:max_docs]
        urls_to_try = self.reranker_urls
        last_error = None
        for url in urls_to_try:
            base_url = url.rstrip('/')
            rerank_url = f"{base_url}/rerank"
            request_timeout = self.timeout
            try:
                logger.debug(f"Запрос реранкинга к {rerank_url}: {len(documents)} документов (timeout={request_timeout}s)")
                response = requests.post(
                    rerank_url,
                    json={
                        'query': query,
                        'documents': documents
                    },
                    timeout=request_timeout
                )
                if response.status_code == 200:
                    result = response.json()
                    reranked = result.get('reranked_documents', documents)
                    logger.info(f"Реранкинг успешен: {len(reranked)} документов через {url}")
                    return reranked
                else:
                    logger.warning(f"Ошибка реранкинга: {response.status_code} - {response.text}")
                    last_error = f"HTTP {response.status_code}"
            except requests.exceptions.Timeout:
                logger.warning(f"Таймаут подключения к {rerank_url} ({request_timeout}s)")
                last_error = f"Timeout {request_timeout}s"
                continue
            except requests.exceptions.RequestException as e:
                logger.warning(f"Ошибка подключения к {rerank_url}: {e}")
                last_error = str(e)
                continue
        logger.error(f"Все сервисы реранкинга недоступны. Последняя ошибка: {last_error}")
        return documents
    def health_check(self) -> bool:

        for url in self.reranker_urls:
            url = url.rstrip('/')
            health_url = f"{url}/health"
            try:
                response = requests.get(health_url, timeout=5)
                if response.status_code == 200:
                    return True
            except:
                continue
        return False
reranker_client = RerankerClient()