import os
import sys
import time
import requests
import logging
import subprocess
from pathlib import Path
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)
class RAGSystemManager:

    def __init__(self):
        self.base_dir = Path('/opt/autogen')
        ollama_base_url = os.getenv('OLLAMA_BASE_URL', '')
        self.services = {
            'ollama': {'port': 80, 'url': f'{ollama_base_url.rstrip("/")}/api/tags'},
            'chromadb': {'port': int(os.getenv("CHROMADB_PORT", "8000")), 'url': f'{os.getenv("CHROMADB_BASE_URL", "")}/api/v2/collections'},
            'rag-api': {'port': 9004, 'url': f'http://localhost:{os.getenv("FLASK_PORT", "9004")}/api/status'},
            'analytics': {'port': 9005, 'url': f'http://localhost:{os.getenv("ANALYTICS_PORT", "9005")}/api/stats'}
        }
    def check_service(self, service_name: str) -> bool:

        try:
            if service_name == 'chromadb':
                try:
                    import chromadb
                    from chromadb.config import Settings
                    settings = Settings(anonymized_telemetry=False, allow_reset=True)
                    client = chromadb.HttpClient(
                        host=os.getenv("CHROMADB_HOST", ""),
                        port=int(os.getenv("CHROMADB_PORT", "8000")),
                        settings=settings
                    )
                    client.list_collections()
                    logger.info(f" {service_name} доступен")
                    return True
                except Exception as e:
                    logger.warning(f" {service_name} недоступен: {e}")
                    return False
            else:
                response = requests.get(self.services[service_name]['url'], timeout=5)
            if response.status_code == 200:
                logger.info(f" {service_name} доступен")
                return True
            else:
                logger.warning(f" {service_name} недоступен (код: {response.status_code})")
                return False
        except Exception as e:
            logger.warning(f" {service_name} недоступен: {e}")
            return False
    def wait_for_service(self, service_name: str, max_wait: int = 60) -> bool:

        logger.info(f"Ожидание доступности {service_name}...")
        for i in range(max_wait):
            if self.check_service(service_name):
                return True
            time.sleep(1)
        logger.error(f" {service_name} не стал доступен за {max_wait} секунд")
        return False
    def start_docker_services(self) -> bool:

        logger.info("Запуск Docker сервисов...")
        try:
            os.chdir(self.base_dir)
            result = subprocess.run(
                ['docker-compose', 'up', '-d'],
                capture_output=True,
                text=True,
                timeout=300
            )
            if result.returncode == 0:
                logger.info(" Docker сервисы запущены")
                return True
            else:
                logger.error(f" Ошибка запуска Docker сервисов: {result.stderr}")
                return False
        except Exception as e:
            logger.error(f" Ошибка запуска Docker сервисов: {e}")
            return False
    def stop_docker_services(self) -> bool:

        logger.info("Остановка Docker сервисов...")
        try:
            os.chdir(self.base_dir)
            result = subprocess.run(
                ['docker-compose', 'down'],
                capture_output=True,
                text=True,
                timeout=60
            )
            if result.returncode == 0:
                logger.info(" Docker сервисы остановлены")
                return True
            else:
                logger.error(f" Ошибка остановки Docker сервисов: {result.stderr}")
                return False
        except Exception as e:
            logger.error(f" Ошибка остановки Docker сервисов: {e}")
            return False
    def process_documents(self) -> bool:

        logger.info("Обработка документов...")
        try:
            result = subprocess.run(
                [sys.executable, 'app/process_documents.py'],
                capture_output=True,
                text=True,
                timeout=600
            )
            if result.returncode == 0:
                logger.info(" Документы обработаны")
                return True
            else:
                logger.error(f" Ошибка обработки документов: {result.stderr}")
                return False
        except Exception as e:
            logger.error(f" Ошибка обработки документов: {e}")
            return False
    def test_chat_api(self) -> bool:

        logger.info("Тестирование Chat API...")
        try:
            test_data = {
                "question": "Что такое донорство крови?",
                "temperature": 0.05,
                "search_k": 100
            }
            response = requests.post(
                f'http://localhost:{os.getenv("FLASK_PORT", "9004")}/api/chat/stream',
                json=test_data,
                timeout=30
            )
            if response.status_code == 200:
                result = response.json()
                logger.info(f" Chat API работает. Ответ: {result['response'][:100]}...")
                return True
            else:
                logger.error(f" Ошибка Chat API: {response.status_code}")
                return False
        except Exception as e:
            logger.error(f" Ошибка тестирования Chat API: {e}")
            return False
    def show_status(self):

        logger.info("Проверка статуса сервисов...")
        for service_name in self.services:
            self.check_service(service_name)
    def show_logs(self, service_name: str = None):

        try:
            if service_name:
                cmd = ['docker-compose', 'logs', '-f', '--tail=50', service_name]
            else:
                cmd = ['docker-compose', 'logs', '-f', '--tail=50']
            subprocess.run(cmd)
        except Exception as e:
            logger.error(f" Ошибка показа логов: {e}")
def main():

    if len(sys.argv) < 2:
        print("Использование:")
        print("  python run_system.py start    - Запустить систему")
        print("  python run_system.py stop     - Остановить систему")
        print("  python run_system.py status   - Показать статус")
        print("  python run_system.py test     - Тестировать API")
        print("  python run_system.py logs     - Показать логи")
        print("  python run_system.py process  - Обработать документы")
        sys.exit(1)
    command = sys.argv[1]
    manager = RAGSystemManager()
    if command == 'start':
        logger.info(" Запуск RAG Chat API системы...")
        if not manager.start_docker_services():
            sys.exit(1)
        if not manager.wait_for_service('ollama', 60):
            sys.exit(1)
        if not manager.wait_for_service('chromadb', 60):
            sys.exit(1)
        if not manager.wait_for_service('analytics', 60):
            sys.exit(1)
        if not manager.wait_for_service('rag-api', 60):
            sys.exit(1)
        if not manager.process_documents():
            logger.warning(" Ошибка обработки документов, но система запущена")
        logger.info(" Обновление конфигурации реверс-прокси...")
        try:
            subprocess.run(['/opt/autogen/update_reverse_proxy.sh'], check=True, timeout=30)
            logger.info(" Реверс-прокси обновлен")
        except Exception as e:
            logger.warning(f" Не удалось обновить реверс-прокси: {e}")
        logger.info(" Система запущена и готова к работе!")
        logger.info("🌐 Веб-интерфейс: https://your-rag-chat-api.com")
        logger.info("🎯 Виджет демо: https://your-rag-chat-api.com/widget")
        logger.info("📊 API статус: https://your-rag-chat-api.com/api/status")
        logger.info("🎯 Demo сайт: https://rag-chat-demo.your-domain.com")
        logger.info("📈 Аналитика: https://your-analytics.com")
    elif command == 'stop':
        logger.info("🛑 Остановка RAG Chat API системы...")
        manager.stop_docker_services()
    elif command == 'status':
        manager.show_status()
    elif command == 'test':
        if manager.test_chat_api():
            logger.info(" Тест прошел успешно")
        else:
            logger.error(" Тест не прошел")
            sys.exit(1)
    elif command == 'logs':
        service = sys.argv[2] if len(sys.argv) > 2 else None
        manager.show_logs(service)
    elif command == 'process':
        if manager.process_documents():
            logger.info(" Документы обработаны")
        else:
            logger.error(" Ошибка обработки документов")
            sys.exit(1)
    else:
        logger.error(f" Неизвестная команда: {command}")
        sys.exit(1)
if __name__ == "__main__":
    main()