(function () {
    'use strict';

    const WIDGET_CONFIG = {
        apiUrl: window.RAG_CHAT_API_URL || getApiUrl(),
        position: 'bottom-right',
        theme: 'default',
        language: 'ru',
        autoOpen: false,
        showPoweredBy: false,

        modelSettings: {
            temperature: 0.2,
            search_k: 100,
            model: 'qwen3:30b',
            embedding_model: 'jeffh/intfloat-multilingual-e5-large-instruct:q8_0',
            num_ctx: 50000,
            timeout: 600
        }
    };

    function getApiUrl() {
        try {

            const script = document.currentScript || document.querySelector('script[src*="widget.js"]');
            if (script) {
                const explicitApiUrl = script.getAttribute('data-api-url');
                if (explicitApiUrl) return explicitApiUrl;
            }

            if (script && script.src) {
                const scriptUrl = new URL(script.src, window.location.href);
                if (scriptUrl.origin) return scriptUrl.origin;
            }
        } catch (_) {  }

        if (window.location.protocol === 'file:') {
            return window.WIDGET_API_URL || window.location.origin || 'http://localhost:9008';
        }

        return window.location.protocol + '//' + window.location.host;
    }

    async function loadModelSettings() {
        try {
            const response = await fetch(`${WIDGET_CONFIG.apiUrl}/api/widget/config`);
            if (response.ok) {
                const settings = await response.json();
                WIDGET_CONFIG.modelSettings = { ...WIDGET_CONFIG.modelSettings, ...settings };
                console.log('Настройки модели загружены:', WIDGET_CONFIG.modelSettings);
            }
        } catch (error) {
            console.warn('Не удалось загрузить настройки модели, используются значения по умолчанию:', error);
        }
    }

    const WIDGET_CSS = `

        :root {
            --sk-primary: #EE2E22;
            --sk-primary-light: #f25b51;
            --sk-primary-dark: #cd1b10;
            --sk-secondary: #F1633F;
            --sk-bg: #fff;
            --sk-text: #0C0C0C;
            --sk-gradient-start: #ffffff;
            --sk-gradient-mid1: #fff5f4;
            --sk-gradient-mid2: #ffe0dd;
        }

        body.dark-theme,
        body.theme-dark,
        body[data-theme="dark"],
        html.dark-theme,
        html[data-theme="dark"],
        .bvi-body,
        .rag-chat-widget.dark-theme {
            --sk-bg: #1a1a1a;
            --sk-text: #f0f0f0;
            --sk-gradient-start: #1a1a1a;
            --sk-gradient-mid1: #2a1a1a;
            --sk-gradient-mid2: #3a1515;
        }

        .rag-chat-widget.dark-theme .rag-chat-header h3 {
            color: #fff;
        }

        .rag-chat-widget.dark-theme .rag-chat-input-container {
            background: rgba(30, 30, 30, 0.9);
            border-top-color: rgba(238, 46, 34, 0.3);
        }

        .rag-chat-widget.dark-theme .rag-chat-input {
            background: #2a2a2a;
            color: #f0f0f0;
            border-color: rgba(238, 46, 34, 0.3);
        }

        .rag-chat-widget.dark-theme .rag-chat-powered-by {
            background: rgba(30, 30, 30, 0.9);
            color: #888;
            border-top-color: rgba(238, 46, 34, 0.3);
        }

        .rag-chat-widget {
            position: fixed;
            z-index: 999999;
            font-family: 'TT Hoves', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            font-size: 14px;
            line-height: 1.5;
            color: var(--sk-text);
        }

        .rag-chat-widget.bottom-right {
            bottom: 20px;
            right: 20px;
        }

        .rag-chat-widget.bottom-left {
            bottom: 20px;
            left: 20px;
        }

        .rag-chat-widget.top-right {
            top: 20px;
            right: 20px;
        }

        .rag-chat-widget.top-left {
            top: 20px;
            left: 20px;
        }

        .rag-chat-toggle {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: var(--sk-primary);
            border: none;
            cursor: pointer;
            box-shadow: 0 4px 16px rgba(238, 46, 34, 0.35);
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .rag-chat-toggle:hover {
            transform: scale(1.1);
            box-shadow: 0 6px 24px rgba(238, 46, 34, 0.45);
            background: var(--sk-primary-dark);
        }

        .rag-chat-toggle:active {
            transform: scale(0.95);
        }

        .rag-chat-toggle-icon {
            width: 24px;
            height: 24px;
            fill: white;
            transition: transform 0.3s ease;
        }

        .rag-chat-toggle.open .rag-chat-toggle-icon {
            transform: rotate(45deg);
        }

        .rag-chat-widget.dark-theme .rag-chat-toggle {
            background: var(--sk-primary);
            box-shadow: 0 4px 16px rgba(238, 46, 34, 0.5);
        }

        .rag-chat-widget.dark-theme .rag-chat-toggle:hover {
            background: var(--sk-primary-light);
            box-shadow: 0 6px 24px rgba(238, 46, 34, 0.6);
        }

        .rag-chat-toggle-badge {
            position: absolute;
            top: -5px;
            right: -5px;
            background: var(--sk-secondary);
            color: white;
            border-radius: 50%;
            width: 20px;
            height: 20px;
            font-size: 12px;
            font-weight: bold;
            display: flex;
            align-items: center;
            justify-content: center;
            animation: pulse 2s infinite;
        }

        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.1); }
            100% { transform: scale(1); }
        }

        .rag-chat-window {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            width: 100vw;
            height: 100vh;
            background: linear-gradient(180deg, var(--sk-gradient-start) 0%, var(--sk-gradient-mid1) 30%, var(--sk-gradient-mid2) 60%, var(--sk-primary) 100%);
            border-radius: 0;
            box-shadow: none;
            display: none; 
            flex-direction: column;
            overflow: hidden;
            animation: fadeIn 0.3s ease;
            z-index: 999998;
        }

        .rag-chat-window.open {
            display: flex;
        }

        .rag-chat-window.open {
            display: flex;
        }

        @keyframes fadeIn {
            from {
                opacity: 0;
            }
            to {
                opacity: 1;
            }
        }

        .rag-chat-header {
            position: relative;
            background: transparent;
            color: var(--sk-primary);
            padding: 20px 30px;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .rag-chat-header h3 {
            margin: 0;
            font-size: 24px;
            font-weight: 700;
            color: var(--sk-primary);
        }

        .rag-chat-header-controls {
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .rag-chat-close {
            background: var(--sk-primary);
            border: none;
            color: white;
            cursor: pointer;
            font-size: 24px;
            padding: 0;
            width: 44px;
            height: 44px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            transition: all 0.2s ease;
            box-shadow: 0 4px 12px rgba(238, 46, 34, 0.3);
        }

        .rag-chat-close:hover {
            background: var(--sk-primary-dark);
            transform: scale(1.1);
            box-shadow: 0 6px 20px rgba(238, 46, 34, 0.4);
        }

        .rag-chat-messages {
            flex: 1;
            padding: 30px;
            padding-top: 20px;
            padding-bottom: 30px;
            overflow-y: auto;
            background: transparent;
            max-width: 900px;
            margin: 0 auto;
            width: 100%;
        }

        .rag-chat-message {
            margin-bottom: 16px;
            display: flex;
            flex-direction: column;
            gap: 6px;
        }

        .rag-chat-message.user {
            align-items: flex-end;
        }

        .rag-chat-message.assistant {
            align-items: flex-start;
        }

        .rag-chat-message-content {
            max-width: 70%;
            padding: 16px 24px;
            border-radius: 20px;
            word-wrap: break-word;
            white-space: pre-wrap;
            font-size: 16px;
            line-height: 24px;
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.08);
        }

        .rag-chat-message.user .rag-chat-message-content {
            background: var(--sk-primary);
            color: white;
            border-bottom-right-radius: 4px;
        }

        .rag-chat-message.assistant .rag-chat-message-content {
            background: var(--sk-bg);
            color: var(--sk-text);
            border-bottom-left-radius: 4px;
            border: 1px solid rgba(238, 46, 34, 0.1);
        }

        body.dark-theme .rag-chat-message.assistant .rag-chat-message-content,
        body.theme-dark .rag-chat-message.assistant .rag-chat-message-content,
        body[data-theme="dark"] .rag-chat-message.assistant .rag-chat-message-content,
        .bvi-body .rag-chat-message.assistant .rag-chat-message-content {
            background: rgba(30, 30, 30, 0.95);
            border-color: rgba(238, 46, 34, 0.3);
        }

        .rag-welcome-message {
            padding: 0 !important;
        }

        .rag-welcome-content {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 20px;
            padding: 20px 24px;
            text-align: center;
        }

        .rag-welcome-image {
            width: 224px;
            height: 224px;
            border-radius: 16px;
            object-fit: cover;
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.15);
        }

        .rag-welcome-text {
            line-height: 1.6;
            max-width: 600px;
        }

        @media (max-width: 480px) {
            .rag-welcome-content {
                gap: 16px;
                padding: 16px;
            }
            .rag-welcome-image {
                width: 120px;
                height: 120px;
            }
        }

        .rag-chat-message-content a {
            color: var(--sk-primary);
            text-decoration: underline;
            transition: color 0.2s ease;
        }

        .rag-chat-message.user .rag-chat-message-content a {
            color: #fff;
        }

        .rag-chat-message-content a:hover {
            opacity: 0.8;
        }

        .rag-chat-typing {
            display: none;
            align-items: center;
            color: #666;
            font-style: italic;
            margin-bottom: 12px;
        }

        .rag-chat-typing.show {
            display: flex;
        }

        .rag-chat-typing-dots {
            display: inline-block;
            margin-left: 5px;
        }

        .rag-chat-typing-dots span {
            display: inline-block;
            width: 5px;
            height: 5px;
            border-radius: 50%;
            background: var(--sk-primary);
            margin: 0 2px;
            animation: typing 1.4s infinite ease-in-out;
        }

        .rag-chat-typing-dots span:nth-child(1) { animation-delay: -0.32s; }
        .rag-chat-typing-dots span:nth-child(2) { animation-delay: -0.16s; }

        @keyframes typing {
            0%, 80%, 100% { transform: scale(0); }
            40% { transform: scale(1); }
        }

        .rag-chat-status {
            display: flex;
            align-items: center;
            padding: 10px 14px;
            margin-bottom: 10px;
            border-radius: 10px;
            font-size: 13px;
            color: #4a4a4a;
            background: var(--sk-bg);
            border-left: 3px solid var(--sk-primary);
            animation: fadeIn 0.3s ease;
        }

        .rag-chat-status.searching { border-left-color: #4dabf7; }
        .rag-chat-status.processing { border-left-color: #51cf66; }
        .rag-chat-status.warming_up { border-left-color: var(--sk-secondary); }
        .rag-chat-status.generating { border-left-color: #ffd43b; }
        .rag-chat-status.error {
            border-left-color: var(--sk-primary);
            background: #ffe3e3;
            color: #c92a2a;
        }
        .rag-chat-status.complete { border-left-color: #20c997; }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(6px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .rag-chat-input-container {
            padding: 16px 20px;
            background: var(--sk-bg);
            border-top: 1px solid #EDEEF1;
            display: flex;
            justify-content: center;
            box-sizing: border-box;
        }

        .rag-chat-input-form {
            display: flex;
            gap: 0;
            width: 100%;
            max-width: 800px;
            box-sizing: border-box;
        }

        .rag-chat-input {
            flex: 1;
            min-width: 0;
            padding: 11px 15px;
            border: 2px solid var(--sk-primary);
            border-right: none;
            border-top-left-radius: 12px;
            border-bottom-left-radius: 12px;
            border-top-right-radius: 0;
            border-bottom-right-radius: 0;
            font-size: 13px;
            outline: none;
            transition: border-color 0.2s ease;
            box-sizing: border-box;
        }

        .rag-chat-input:focus {
            border-color: var(--sk-primary-dark);
        }

        .rag-chat-input::placeholder {
            color: #9DA2A8;
        }

        .rag-chat-send {
            padding: 11px 18px;
            background: var(--sk-primary);
            color: white;
            border: 2px solid var(--sk-primary);
            border-top-right-radius: 12px;
            border-bottom-right-radius: 12px;
            border-top-left-radius: 0;
            border-bottom-left-radius: 0;
            cursor: pointer;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.2s ease;
            flex-shrink: 0;
            white-space: nowrap;
            box-sizing: border-box;
        }

        .rag-chat-send:hover {
            background: var(--sk-primary-dark);
            border-color: var(--sk-primary-dark);
        }

        .rag-chat-send:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }

        .rag-chat-powered-by {
            text-align: center;
            padding: 10px;
            font-size: 11px;
            color: #9DA2A8;
            border-top: 1px solid #EDEEF1;
            background: var(--sk-bg);
        }

        .rag-chat-powered-by a {
            color: var(--sk-primary);
            text-decoration: none;
        }

        .rag-chat-powered-by a:hover {
            text-decoration: underline;
        }

        @media (max-width: 480px) {
            .rag-chat-messages {
                padding: 15px;
                max-width: 100%;
            }
            .rag-chat-message-content {
                max-width: 90%;
                font-size: 14px;
            }
            .rag-chat-header h3 {
                font-size: 18px;
            }
            .rag-chat-close {
                width: 36px;
                height: 36px;
                font-size: 20px;
            }
        }

        @media (min-width: 768px) {
            .rag-chat-messages {
                max-width: 700px;
            }
            .rag-chat-message-content {
                max-width: 75%;
            }
            .rag-welcome-image {
                width: 180px;
                height: 180px;
            }
        }

        @media (min-width: 1024px) {
            .rag-chat-messages {
                max-width: 800px;
                padding: 40px;
            }
            .rag-chat-message-content {
                padding: 18px 24px;
                max-width: 70%;
                font-size: 16px;
                line-height: 24px;
            }
            .rag-welcome-image {
                width: 224px;
                height: 224px;
            }
        }

        .rag-accessibility-btn {
            background: var(--sk-primary);
            border: none;
            color: white;
            cursor: pointer;
            font-size: 16px;
            padding: 8px 12px;
            margin-right: 8px;
            border-radius: 8px;
            transition: all 0.2s ease;
            display: flex;
            align-items: center;
            gap: 6px;
            box-shadow: 0 2px 8px rgba(238, 46, 34, 0.3);
        }

        .rag-accessibility-btn:hover {
            background: var(--sk-primary-dark);
            transform: scale(1.05);
        }

        .rag-accessibility-btn svg {
            width: 20px;
            height: 20px;
        }

        .rag-accessibility-panel {
            display: none;
            position: absolute;
            top: calc(100% + 10px);
            right: 0;
            min-width: 280px;
            background: #1a1a1a;
            color: white;
            padding: 16px;
            z-index: 10000;
            box-shadow: 0 8px 24px rgba(0,0,0,0.4);
            border-radius: 12px;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .rag-chat-widget.dark-theme .rag-accessibility-panel {
            background: #2a2a2a;
            border: 1px solid rgba(238, 46, 34, 0.3);
        }

        .rag-accessibility-panel.active {
            display: block;
        }

        .rag-accessibility-row {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 10px;
            padding-bottom: 10px;
            border-bottom: 1px solid #333;
        }

        .rag-accessibility-row:last-child {
            margin-bottom: 0;
            padding-bottom: 0;
            border-bottom: none;
        }

        .rag-accessibility-label {
            font-size: 12px;
            font-weight: 500;
        }

        .rag-accessibility-controls {
            display: flex;
            gap: 4px;
        }

        .rag-accessibility-controls button {
            background: #333;
            border: 1px solid #555;
            color: white;
            padding: 4px 10px;
            cursor: pointer;
            border-radius: 3px;
            font-size: 12px;
            transition: all 0.2s;
        }

        .rag-accessibility-controls button:hover {
            background: #444;
        }

        .rag-accessibility-controls button.active {
            background: var(--sk-primary);
            border-color: var(--sk-primary);
        }

        .rag-chat-widget.vi-font-large .rag-chat-message-content {
            font-size: 18px !important;
            line-height: 26px !important;
        }

        .rag-chat-widget.vi-font-xlarge .rag-chat-message-content {
            font-size: 22px !important;
            line-height: 32px !important;
        }

        .rag-chat-widget.vi-font-large .rag-chat-input,
        .rag-chat-widget.vi-font-large .rag-chat-send {
            font-size: 16px !important;
        }

        .rag-chat-widget.vi-font-xlarge .rag-chat-input,
        .rag-chat-widget.vi-font-xlarge .rag-chat-send {
            font-size: 20px !important;
        }

        .rag-chat-widget.vi-font-large .rag-chat-header h3 {
            font-size: 16px !important;
        }

        .rag-chat-widget.vi-font-xlarge .rag-chat-header h3 {
            font-size: 20px !important;
        }

        .rag-chat-widget.vi-contrast-wb .rag-chat-messages {
            background: #000 !important;
        }
        .rag-chat-widget.vi-contrast-wb .rag-chat-message-content {
            background: #000 !important;
            color: #fff !important;
            border: 2px solid #fff !important;
        }
        .rag-chat-widget.vi-contrast-wb .rag-chat-input-container {
            background: #000 !important;
            border-top: 2px solid #fff !important;
        }
        .rag-chat-widget.vi-contrast-wb .rag-chat-input {
            background: #000 !important;
            color: #fff !important;
            border: 2px solid #fff !important;
        }
        .rag-chat-widget.vi-contrast-wb .rag-chat-send {
            background: #fff !important;
            color: #000 !important;
        }
        .rag-chat-widget.vi-contrast-wb .rag-chat-header {
            background: #000 !important;
            border-bottom: 2px solid #fff !important;
        }
        .rag-chat-widget.vi-contrast-wb .rag-chat-window {
            border: 2px solid #fff !important;
        }

        .rag-chat-widget.vi-contrast-bw .rag-chat-messages {
            background: #fff !important;
        }
        .rag-chat-widget.vi-contrast-bw .rag-chat-message-content {
            background: #fff !important;
            color: #000 !important;
            border: 2px solid #000 !important;
        }
        .rag-chat-widget.vi-contrast-bw .rag-chat-input-container {
            background: #fff !important;
            border-top: 2px solid #000 !important;
        }
        .rag-chat-widget.vi-contrast-bw .rag-chat-input {
            background: #fff !important;
            color: #000 !important;
            border: 2px solid #000 !important;
        }
        .rag-chat-widget.vi-contrast-bw .rag-chat-send {
            background: #000 !important;
            color: #fff !important;
        }
        .rag-chat-widget.vi-contrast-bw .rag-chat-header {
            background: #fff !important;
            color: #000 !important;
            border-bottom: 2px solid #000 !important;
        }
        .rag-chat-widget.vi-contrast-bw .rag-chat-header h3,
        .rag-chat-widget.vi-contrast-bw .rag-chat-close,
        .rag-chat-widget.vi-contrast-bw .rag-accessibility-btn {
            color: #000 !important;
        }
        .rag-chat-widget.vi-contrast-bw .rag-chat-window {
            border: 2px solid #000 !important;
        }

        .rag-chat-widget.vi-contrast-blue .rag-chat-messages {
            background: #9dd1ff !important;
        }
        .rag-chat-widget.vi-contrast-blue .rag-chat-message-content {
            background: #063462 !important;
            color: #fff !important;
        }
        .rag-chat-widget.vi-contrast-blue .rag-chat-input-container {
            background: #063462 !important;
        }
        .rag-chat-widget.vi-contrast-blue .rag-chat-input {
            background: #fff !important;
            color: #063462 !important;
        }
        .rag-chat-widget.vi-contrast-blue .rag-chat-header {
            background: #063462 !important;
        }
        .rag-chat-widget.vi-contrast-blue .rag-chat-send {
            background: #9dd1ff !important;
            color: #063462 !important;
        }
    `;

    class RAGChatWidget {
        constructor(config = {}) {
            console.log(' Создание экземпляра RAGChatWidget с конфигурацией:', config);

            this.config = { ...WIDGET_CONFIG, ...config };
            this.isOpen = false; 
            this.conversationId = this.generateStableConversationId();
            this.messages = [];
            this.isLoading = false;
            this.statusElements = [];

            console.log(' Инициализация виджета...');
            this.init();
            console.log(' Виджет инициализирован');
        }

        generateStableConversationId() {

            const browserFingerprint = navigator.userAgent + navigator.language + screen.width + screen.height;
            const date = new Date().toISOString().split('T')[0]; 
            const rawId = browserFingerprint + '_' + date;

            let hash = 0;
            for (let i = 0; i < rawId.length; i++) {
                const char = rawId.charCodeAt(i);
                hash = ((hash << 5) - hash) + char;
                hash = hash & hash; 
            }
            return Math.abs(hash).toString(16).substring(0, 16);
        }

        async init() {

            await loadModelSettings();

            this.injectCSS();
            this.createWidget();
            this.bindEvents();
        }

        injectCSS() {
            if (document.getElementById('rag-chat-widget-css')) return;

            const style = document.createElement('style');
            style.id = 'rag-chat-widget-css';
            style.textContent = WIDGET_CSS;
            document.head.appendChild(style);
        }

        createWidget() {

            this.container = document.createElement('div');
            this.container.className = `rag-chat-widget ${this.config.position}`;

            this.toggleButton = document.createElement('button');
            this.toggleButton.className = 'rag-chat-toggle';
            this.toggleButton.innerHTML = `
                <svg class="rag-chat-toggle-icon" viewBox="0 0 24 24">
                    <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm5 11h-4v4h-2v-4H7v-2h4V7h2v4h4v2z"/>
                </svg>
            `;

            this.chatWindow = document.createElement('div');
            this.chatWindow.className = 'rag-chat-window';
            this.chatWindow.innerHTML = `
                <div class="rag-chat-header">
                    <h3>Помощник донора</h3>
                    <div class="rag-chat-header-controls">
                        <button class="rag-accessibility-btn" id="rag-accessibility-toggle" title="Версия для слабовидящих">
                            <svg viewBox="0 0 24 24" fill="currentColor"><path d="M12 4.5C7 4.5 2.73 7.61 1 12c1.73 4.39 6 7.5 11 7.5s9.27-3.11 11-7.5c-1.73-4.39-6-7.5-11-7.5zM12 17c-2.76 0-5-2.24-5-5s2.24-5 5-5 5 2.24 5 5-2.24 5-5 5zm0-8c-1.66 0-3 1.34-3 3s1.34 3 3 3 3-1.34 3-3-1.34-3-3-3z"/></svg>
                        </button>
                        <button class="rag-chat-close">×</button>
                    </div>
                    <div class="rag-accessibility-panel" id="rag-accessibility-panel">
                        <div class="rag-accessibility-row">
                            <span class="rag-accessibility-label">Размер шрифта</span>
                            <div class="rag-accessibility-controls" id="rag-font-controls">
                                <button data-font="normal" class="active">A</button>
                                <button data-font="large">A+</button>
                                <button data-font="xlarge">A++</button>
                            </div>
                        </div>
                        <div class="rag-accessibility-row">
                            <span class="rag-accessibility-label">Цветовая схема</span>
                            <div class="rag-accessibility-controls" id="rag-contrast-controls">
                                <button data-contrast="normal" class="active">Обычная</button>
                                <button data-contrast="wb">Бел/Чёр</button>
                                <button data-contrast="bw">Чёр/Бел</button>
                                <button data-contrast="blue">Синяя</button>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="rag-chat-messages" id="rag-chat-messages">
                    <div class="rag-chat-message assistant">
                        <div class="rag-chat-message-content rag-welcome-message">
                            <div class="rag-welcome-content">
                                <img src="" alt="Помощник донора" class="rag-welcome-image" id="rag-welcome-img">
                                <div class="rag-welcome-text">
Здравствуйте! Я помощник Службы крови. Отвечу на вопросы о донорстве: как стать донором, требования к здоровью, противопоказания, льготы и выплаты донорам. Спрашивайте!
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="rag-chat-typing" id="rag-chat-typing">
                    <span>Бот печатает</span>
                    <div class="rag-chat-typing-dots">
                        <span></span>
                        <span></span>
                        <span></span>
                    </div>
                </div>
                <div class="rag-chat-input-container">
                    <form class="rag-chat-input-form" id="rag-chat-form">
                        <input type="text" class="rag-chat-input" id="rag-chat-input" placeholder="Задайте вопрос..." autocomplete="off">
                        <button type="submit" class="rag-chat-send" id="rag-chat-send">Отправить</button>
                    </form>
                </div>
                ${this.config.showPoweredBy ? `
                    <div class="rag-chat-powered-by">
                        Powered by <a href="https://your-rag-chat-api.com" target="_blank">Тёмыч дримтим</a>
                    </div>
                ` : ''}
            `;

            this.container.appendChild(this.toggleButton);
            this.container.appendChild(this.chatWindow);
            document.body.appendChild(this.container);

            this.messagesContainer = this.chatWindow.querySelector('#rag-chat-messages');
            this.typingIndicator = this.chatWindow.querySelector('#rag-chat-typing');
            this.inputForm = this.chatWindow.querySelector('#rag-chat-form');
            this.inputField = this.chatWindow.querySelector('#rag-chat-input');
            this.sendButton = this.chatWindow.querySelector('#rag-chat-send');
            this.closeButton = this.chatWindow.querySelector('.rag-chat-close');

            const welcomeImg = this.chatWindow.querySelector('#rag-welcome-img');
            if (welcomeImg) {
                const apiUrl = this.config.apiUrl || getApiUrl();

                const baseUrl = apiUrl.replace(/\/$/, '');
                welcomeImg.src = `${baseUrl}/static/images/14.png`;
            }
        }

        bindEvents() {
            this.toggleButton.addEventListener('click', () => this.toggle());
            this.closeButton.addEventListener('click', () => this.close());
            this.inputForm.addEventListener('submit', (e) => this.handleSubmit(e));

            this.setupAccessibility();

            this.setupThemeInheritance();
        }

        setupThemeInheritance() {

            const checkDarkTheme = () => {
                const body = document.body;
                const html = document.documentElement;

                let isDark = false;

                const bodyClasses = ['dark-theme', 'theme-dark', 'bvi-body', 'dark', 'dark-mode', 'night-mode'];
                for (let cls of bodyClasses) {
                    if (body.classList.contains(cls)) {
                        isDark = true;
                        break;
                    }
                }

                if (!isDark) {
                    const bodyTheme = body.getAttribute('data-theme') || body.getAttribute('data-color-mode') || body.getAttribute('data-mode');
                    if (bodyTheme === 'dark' || bodyTheme === 'night') {
                        isDark = true;
                    }
                }

                if (!isDark) {
                    for (let cls of bodyClasses) {
                        if (html.classList.contains(cls)) {
                            isDark = true;
                            break;
                        }
                    }
                }

                if (!isDark) {
                    const htmlTheme = html.getAttribute('data-theme') || html.getAttribute('data-color-mode') || html.getAttribute('data-mode');
                    if (htmlTheme === 'dark' || htmlTheme === 'night') {
                        isDark = true;
                    }
                }

                if (!isDark) {
                    const colorScheme = window.getComputedStyle(body).getPropertyValue('color-scheme');
                    if (colorScheme && colorScheme.trim() === 'dark') {
                        isDark = true;
                    }
                }

                if (!isDark) {
                    const storedTheme = localStorage.getItem('theme') || localStorage.getItem('color-theme') || localStorage.getItem('colorMode');
                    if (storedTheme === 'dark' || storedTheme === 'night') {
                        isDark = true;
                    }
                }

                if (!isDark) {
                    const bgColor = window.getComputedStyle(body).backgroundColor;
                    if (bgColor) {

                        const rgb = bgColor.match(/\d+/g);
                        if (rgb && rgb.length >= 3) {
                            const brightness = (parseInt(rgb[0]) + parseInt(rgb[1]) + parseInt(rgb[2])) / 3;
                            if (brightness < 128) { 
                                isDark = true;
                            }
                        }
                    }
                }

                if (isDark) {
                    this.container.classList.add('dark-theme');
                    this.chatWindow.classList.add('dark-theme');
                } else {
                    this.container.classList.remove('dark-theme');
                    this.chatWindow.classList.remove('dark-theme');
                }
            };

            setTimeout(checkDarkTheme, 100);
            setTimeout(checkDarkTheme, 500);

            const observer = new MutationObserver((mutations) => {
                let shouldCheck = false;
                mutations.forEach(mutation => {
                    if (mutation.type === 'attributes' && 
                        (mutation.attributeName === 'class' || mutation.attributeName === 'data-theme' || mutation.attributeName === 'data-color-mode')) {
                        shouldCheck = true;
                    }
                });
                if (shouldCheck) {
                    checkDarkTheme();
                }
            });

            observer.observe(document.body, { 
                attributes: true, 
                attributeFilter: ['class', 'data-theme', 'data-color-mode'],
                subtree: false
            });
            observer.observe(document.documentElement, { 
                attributes: true, 
                attributeFilter: ['class', 'data-theme', 'data-color-mode'],
                subtree: false
            });

            window.addEventListener('storage', (e) => {
                if (e.key === 'theme' || e.key === 'color-theme') {
                    checkDarkTheme();
                }
            });

            setInterval(checkDarkTheme, 2000);
        }

        setupAccessibility() {

            const initAccessibility = () => {
                const toggleBtn = this.chatWindow.querySelector('#rag-accessibility-toggle');
                const panel = this.chatWindow.querySelector('#rag-accessibility-panel');
                const fontControls = this.chatWindow.querySelector('#rag-font-controls');
                const contrastControls = this.chatWindow.querySelector('#rag-contrast-controls');

                if (!toggleBtn || !panel || !fontControls || !contrastControls) {
                    console.warn('RAG Chat: Accessibility elements not found, retrying...');
                    setTimeout(initAccessibility, 100);
                    return;
                }

                console.log('RAG Chat: Accessibility elements found, initializing...');

                this.loadAccessibilitySettings();

                toggleBtn.addEventListener('click', (e) => {
                    e.stopPropagation();
                    e.preventDefault();
                    console.log('RAG Chat: Accessibility toggle clicked');
                    panel.classList.toggle('active');
                });

                const closePanelHandler = (e) => {
                    if (!panel.contains(e.target) && e.target !== toggleBtn && !toggleBtn.contains(e.target)) {
                        panel.classList.remove('active');
                    }
                };
                document.addEventListener('click', closePanelHandler);

                fontControls.querySelectorAll('button').forEach(btn => {
                    btn.addEventListener('click', (e) => {
                        e.stopPropagation();
                        e.preventDefault();
                        fontControls.querySelectorAll('button').forEach(b => b.classList.remove('active'));
                        btn.classList.add('active');
                        this.setFontSize(btn.dataset.font);
                    });
                });

                contrastControls.querySelectorAll('button').forEach(btn => {
                    btn.addEventListener('click', (e) => {
                        e.stopPropagation();
                        e.preventDefault();
                        contrastControls.querySelectorAll('button').forEach(b => b.classList.remove('active'));
                        btn.classList.add('active');
                        this.setContrast(btn.dataset.contrast);
                    });
                });
            };

            initAccessibility();
        }

        setFontSize(size) {
            this.container.classList.remove('vi-font-large', 'vi-font-xlarge');
            if (size === 'large') {
                this.container.classList.add('vi-font-large');
            } else if (size === 'xlarge') {
                this.container.classList.add('vi-font-xlarge');
            }
            localStorage.setItem('rag-vi-font', size);
        }

        setContrast(contrast) {
            this.container.classList.remove('vi-contrast-wb', 'vi-contrast-bw', 'vi-contrast-blue');
            if (contrast !== 'normal') {
                this.container.classList.add('vi-contrast-' + contrast);
            }
            localStorage.setItem('rag-vi-contrast', contrast);
        }

        loadAccessibilitySettings() {
            const font = localStorage.getItem('rag-vi-font') || 'normal';
            const contrast = localStorage.getItem('rag-vi-contrast') || 'normal';

            this.setFontSize(font);
            this.setContrast(contrast);

            setTimeout(() => {
                const fontControls = this.chatWindow.querySelector('#rag-font-controls');
                const contrastControls = this.chatWindow.querySelector('#rag-contrast-controls');

                if (fontControls) {
                    fontControls.querySelectorAll('button').forEach(b => b.classList.remove('active'));
                    const fontBtn = fontControls.querySelector(`[data-font="${font}"]`);
                    if (fontBtn) fontBtn.classList.add('active');
                }

                if (contrastControls) {
                    contrastControls.querySelectorAll('button').forEach(b => b.classList.remove('active'));
                    const contrastBtn = contrastControls.querySelector(`[data-contrast="${contrast}"]`);
                    if (contrastBtn) contrastBtn.classList.add('active');
                }
            }, 100);
        }

        toggle() {
            if (this.isOpen) {
                this.close();
            } else {
                this.open();
            }
        }

        showStatus(status = 'info', message = 'Обработка...') {
            const statusDiv = document.createElement('div');
            statusDiv.className = `rag-chat-status ${status}`;
            statusDiv.textContent = message;
            this.messagesContainer.appendChild(statusDiv);
            this.statusElements.push(statusDiv);
            this.scrollToBottom();
        }

        clearStatuses() {
            this.statusElements.forEach((el) => el.remove());
            this.statusElements = [];
        }

        open() {
            this.isOpen = true;
            this.toggleButton.classList.add('open');
            this.chatWindow.classList.add('open');
            this.inputField.focus();
        }

        close() {
            this.isOpen = false;
            this.toggleButton.classList.remove('open');
            this.chatWindow.classList.remove('open');
        }

        async handleSubmit(e) {
            e.preventDefault();

            if (this.isLoading) return;

            const question = this.inputField.value.trim();
            if (!question) return;

            this.addMessage('user', question);
            this.inputField.value = '';
            this.setLoading(true);

            try {
                const response = await fetch(`${this.config.apiUrl}/api/chat/stream`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        message: question,
                        conversation_id: this.conversationId,
                        temperature: WIDGET_CONFIG.modelSettings.temperature,
                        search_k: WIDGET_CONFIG.modelSettings.search_k,
                        model: WIDGET_CONFIG.modelSettings.model,
                        embedding_model: WIDGET_CONFIG.modelSettings.embedding_model,
                        num_ctx: WIDGET_CONFIG.modelSettings.num_ctx,
                        timeout: WIDGET_CONFIG.modelSettings.timeout
                    })
                });

                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }

                const reader = response.body.getReader();
                const decoder = new TextDecoder();
                let assistantMessage = '';
                let buffer = '';
                let startTime = Date.now();
                let lastEventTime = startTime;
                let thinkingCount = 0;
                let tokenCount = 0;

                console.log('⏱️ Начало обработки стриминга, время:', new Date().toISOString());

                const messageElement = this.addMessage('assistant', '');
                const messageContent = messageElement.querySelector('.rag-chat-message-content');

                const thinkingContainer = document.createElement('div');
                thinkingContainer.className = 'thinking-container';
                thinkingContainer.style.display = 'none';

                const responseContainer = document.createElement('div');
                responseContainer.className = 'response-container';

                messageContent.appendChild(thinkingContainer);
                messageContent.appendChild(responseContainer);

                const forceUpdate = () => {

                    messageContent.style.display = 'none';
                    messageContent.offsetHeight; 
                    messageContent.style.display = '';
                };

                while (true) {
                    const { done, value } = await reader.read();
                    if (done) break;

                    buffer += decoder.decode(value, { stream: true });
                    const lines = buffer.split('\n');

                    buffer = lines.pop() || '';

                    for (const line of lines) {
                        if (line.startsWith('data: ')) {
                            try {
                                const data = JSON.parse(line.slice(6));

                                if (data.type === 'start' && data.conversation_id) {
                                    this.conversationId = data.conversation_id;
                                    console.log(' Начало стриминга, conversation_id:', data.conversation_id);
                                } else if (data.type === 'status') {
                                    this.showStatus(data.status, data.message);

                                    if (data.status === 'error') {
                                        this.setLoading(false);
                                        assistantMessage = data.message || 'Произошла ошибка';
                                        responseContainer.innerHTML = this.formatMessage(assistantMessage);
                                    } else if (data.status === 'complete') {
                                        this.setLoading(false);
                                        this.clearStatuses();
                                    }

                                    console.log('📊 Статус:', data.status, data.message);
                                } else if (data.type === 'token') {
                                    const currentTime = Date.now();
                                    tokenCount++;
                                    const timeSinceStart = (currentTime - startTime) / 1000;
                                    const timeSinceLast = (currentTime - lastEventTime) / 1000;
                                    console.log(` Токен #${tokenCount} (+${timeSinceLast.toFixed(3)}s, total: ${timeSinceStart.toFixed(3)}s): '${data.content}'`);
                                    lastEventTime = currentTime;

                                    if (tokenCount === 1) {
                                        this.clearStatuses();
                                    }

                                    assistantMessage += data.content;

                                    responseContainer.innerHTML = this.formatMessage(assistantMessage);
                                    forceUpdate();
                                    this.scrollToBottom();

                                    await new Promise(resolve => setTimeout(resolve, 10));
                                } else if (data.type === 'thinking') {
                                    const currentTime = Date.now();
                                    thinkingCount++;
                                    const timeSinceStart = (currentTime - startTime) / 1000;
                                    const timeSinceLast = (currentTime - lastEventTime) / 1000;
                                    console.log(`🤔 Thinking #${thinkingCount} (+${timeSinceLast.toFixed(3)}s, total: ${timeSinceStart.toFixed(3)}s): '${data.content}'`);
                                    lastEventTime = currentTime;

                                    if (thinkingContainer.children.length === 0) {
                                        console.log(' Первое thinking событие, отключаем индикатор загрузки');
                                        this.setLoading(false);
                                    }

                                    const thinkingSpan = document.createElement('span');
                                    thinkingSpan.textContent = data.content;
                                    thinkingSpan.style.opacity = '0.3';
                                    thinkingSpan.style.fontStyle = 'italic';
                                    thinkingSpan.style.color = '#666';
                                    thinkingContainer.appendChild(thinkingSpan);
                                    thinkingContainer.style.display = 'block';
                                    forceUpdate();
                                    this.scrollToBottom();
                                } else if (data.type === 'end') {

                                    console.log(' Стриминг завершен, метаданные:', data.metadata);
                                    thinkingContainer.style.display = 'none';
                                    thinkingContainer.innerHTML = '';
                                    this.setLoading(false); 
                                    this.clearStatuses();

                                    break;
                                }
                            } catch (e) {
                                console.warn('Ошибка парсинга JSON:', e, 'Строка:', line);
                            }
                        }
                    }
                }

            } catch (error) {
                console.error('RAG Chat Widget Error:', error);
                this.addMessage('assistant', 'Извините, произошла ошибка при обработке вашего запроса. Попробуйте еще раз.');
                this.setLoading(false);
            }
        }

        addMessage(type, content) {
            const messageDiv = document.createElement('div');
            messageDiv.className = `rag-chat-message ${type}`;

            const contentDiv = document.createElement('div');
            contentDiv.className = 'rag-chat-message-content';

            contentDiv.innerHTML = this.formatMessage(content);

            messageDiv.appendChild(contentDiv);
            this.messagesContainer.appendChild(messageDiv);

            this.scrollToBottom();

            return messageDiv;
        }

        formatMessage(text) {
            if (!text) return '';

            let formatted = text
                .replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/"/g, '&quot;')
                .replace(/'/g, '&#039;');

            // Обрабатываем markdown элементы (важен порядок!)

            // Блоки кода: ```code```
            formatted = formatted.replace(/```([^`]+)```/g, '<pre style="background: #f4f4f4; padding: 10px; border-radius: 5px; overflow-x: auto; margin: 10px 0;"><code>$1</code></pre>');

            // Инлайн код: `code` (до обработки курсива)
            formatted = formatted.replace(/`([^`]+)`/g, '<code style="background: #f4f4f4; padding: 2px 6px; border-radius: 3px; font-family: monospace; font-size: 0.9em;">$1</code>');

            // Жирный текст: **text** или __text__ (до курсива!)
            formatted = formatted.replace(/\*\*([^\*]+)\*\*/g, '<strong>$1</strong>');
            formatted = formatted.replace(/__([^_]+)__/g, '<strong>$1</strong>');

            // Курсив: *text* (после жирного, чтобы не конфликтовать)
            formatted = formatted.replace(/\*([^\*]+)\*/g, '<em>$1</em>');
            formatted = formatted.replace(/(?<!_)_([^_]+)_(?!_)/g, '<em>$1</em>');

            // Зачёркнутый текст: ~~text~~
            formatted = formatted.replace(/~~([^~]+)~~/g, '<del>$1</del>');

            // Заголовки (от большего к меньшему)
            formatted = formatted.replace(/^#### (.+)$/gm, '<h4 style="margin: 8px 0 4px 0; font-size: 1.05em; font-weight: bold;">$1</h4>');
            formatted = formatted.replace(/^### (.+)$/gm, '<h3 style="margin: 10px 0 5px 0; font-size: 1.1em; font-weight: bold;">$1</h3>');
            formatted = formatted.replace(/^## (.+)$/gm, '<h2 style="margin: 12px 0 6px 0; font-size: 1.2em; font-weight: bold;">$1</h2>');
            formatted = formatted.replace(/^# (.+)$/gm, '<h1 style="margin: 15px 0 8px 0; font-size: 1.3em; font-weight: bold;">$1</h1>');

            // Нумерованные списки
            formatted = formatted.replace(/^\s*(\d+)\.\s+(.+)$/gm, '<li style="margin-left: 20px; list-style-type: decimal;">$2</li>');

            // Маркированные списки
            formatted = formatted.replace(/^\s*[-*+]\s+(.+)$/gm, '<li style="margin-left: 20px; list-style-type: disc;">$1</li>');

            // Ссылки в markdown формате: [text](url) (до автолинков!)
            formatted = formatted.replace(/\[([^\]]+)\]\(([^)]+)\)/g, '<a href="$2" target="_blank" rel="noopener noreferrer" style="color: #0066cc; text-decoration: underline; cursor: pointer;">$1</a>');

            // Преобразуем формат "текст (url)" в гиперссылку с синим текстом (ДО удаления <>)
            formatted = formatted.replace(
                /([а-яёА-ЯЁa-zA-Z\s]+)\s*\(&lt;(https?:\/\/[^&]+)&gt;\)/g,
                '<a href="$2" target="_blank" rel="noopener noreferrer" style="color: #0066cc; text-decoration: underline; cursor: pointer;">$1</a>'
            );

            // Преобразуем формат "текст (url)" БЕЗ < > в гиперссылку с синим текстом
            formatted = formatted.replace(
                /([а-яёА-ЯЁa-zA-Z\s]+)\s*\((https?:\/\/[^)]+)\)/g,
                '<a href="$2" target="_blank" rel="noopener noreferrer" style="color: #0066cc; text-decoration: underline; cursor: pointer;">$1</a>'
            );

            // Убираем оставшиеся < > вокруг ссылок (если модель их добавила)
            formatted = formatted.replace(/&lt;(https?:\/\/[^&]+)&gt;/g, '$1');

            // Автоматическое превращение простых URL в ссылки
            formatted = formatted.replace(
                /(?<!href=")(?<!">)(https?:\/\/[^\s<]+[^\s<.,;!?'")\]])/g,
                '<a href="$1" target="_blank" rel="noopener noreferrer" style="color: #0066cc; text-decoration: underline; cursor: pointer;">$1</a>'
            );

            formatted = formatted.replace(/\n/g, '<br>');

            return formatted;
        }

        setLoading(loading) {
            console.log(' setLoading:', loading ? 'ВКЛ' : 'ВЫКЛ');
            this.isLoading = loading;
            this.sendButton.disabled = loading;
            this.typingIndicator.classList.toggle('show', loading);

            if (loading) {
                this.scrollToBottom();
            }
        }

        scrollToBottom() {
            if (this.messagesContainer) {
                this.messagesContainer.scrollTop = this.messagesContainer.scrollHeight;
            }
        }
    }

    async function initWidget() {
        console.log(' Инициализация виджета...');

        if (window.RAGChatWidgetInstance) {
            console.log(' Виджет уже инициализирован');
            return;
        }

        const script = document.querySelector('script[src*="widget.js"]');
        const config = {};

        if (script) {
            const position = script.getAttribute('data-position');
            const theme = script.getAttribute('data-theme');
            const autoOpen = script.getAttribute('data-auto-open');
            const showPoweredBy = script.getAttribute('data-show-powered-by');

            if (position) config.position = position;
            if (theme) config.theme = theme;
            if (autoOpen) config.autoOpen = autoOpen === 'true';
            if (showPoweredBy) config.showPoweredBy = showPoweredBy === 'true';
        }

        console.log(' Конфигурация виджета:', config);

        try {

            window.RAGChatWidgetInstance = new RAGChatWidget(config);
            console.log(' Виджет успешно создан');

            if (config.autoOpen) {
                window.RAGChatWidgetInstance.open();
                console.log('🔓 Виджет автоматически открыт');
            }
        } catch (error) {
            console.error(' Ошибка при создании виджета:', error);
        }
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => initWidget());
    } else {
        initWidget();
    }

    window.RAGChatWidget = RAGChatWidget;

})();